<?php
/**
 * Database operations for Cloudflare Analytics
 * 
 * Tables:
 * - cf_analytics_hourly: Time-series metrics (7 day retention)
 * - cf_analytics_daily: Aggregated daily data (permanent)
 * - cf_analytics_top_urls: Top URLs by day (30 day retention)
 * - cf_analytics_top_countries: Geographic data (30 day retention)
 * - cf_analytics_top_browsers: Browser statistics (30 day retention)
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF_Database {
    
    /**
     * Create all database tables
     */
    public static function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Hourly time-series data (7 days)
        $table_hourly = $wpdb->prefix . 'cf_analytics_hourly';
        $sql_hourly = "CREATE TABLE $table_hourly (
            zone_id varchar(32) NOT NULL,
            datetime datetime NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            unique_visitors int(10) UNSIGNED DEFAULT 0,
            page_views int(10) UNSIGNED DEFAULT 0,
            bandwidth_bytes bigint(20) UNSIGNED DEFAULT 0,
            cached_requests int(10) UNSIGNED DEFAULT 0,
            cached_bytes bigint(20) UNSIGNED DEFAULT 0,
            threats mediumint(8) UNSIGNED DEFAULT 0,
            PRIMARY KEY (zone_id, datetime),
            KEY datetime_idx (datetime)
        ) ENGINE=InnoDB $charset_collate;";
        
        // Daily aggregated data (permanent)
        $table_daily = $wpdb->prefix . 'cf_analytics_daily';
        $sql_daily = "CREATE TABLE $table_daily (
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            unique_visitors int(10) UNSIGNED DEFAULT 0,
            page_views int(10) UNSIGNED DEFAULT 0,
            bandwidth_bytes bigint(20) UNSIGNED DEFAULT 0,
            cached_requests int(10) UNSIGNED DEFAULT 0,
            cached_bytes bigint(20) UNSIGNED DEFAULT 0,
            threats mediumint(8) UNSIGNED DEFAULT 0,
            PRIMARY KEY (zone_id, date),
            KEY date_idx (date)
        ) ENGINE=InnoDB $charset_collate;";
        
        // Top URLs (30 days)
        $table_urls = $wpdb->prefix . 'cf_analytics_top_urls';
        $sql_urls = "CREATE TABLE $table_urls (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            url_path varchar(512) NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            visits int(10) UNSIGNED DEFAULT 0,
            bandwidth_bytes bigint(20) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY zone_date_url (zone_id, date, url_path(255)),
            KEY date_idx (date),
            KEY zone_date_idx (zone_id, date)
        ) ENGINE=InnoDB $charset_collate;";
        
        // Top Countries (30 days)
        $table_countries = $wpdb->prefix . 'cf_analytics_top_countries';
        $sql_countries = "CREATE TABLE $table_countries (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            country_code varchar(2) NOT NULL,
            country_name varchar(100) NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            bandwidth_bytes bigint(20) UNSIGNED DEFAULT 0,
            threats mediumint(8) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY zone_date_country (zone_id, date, country_code),
            KEY date_idx (date),
            KEY zone_date_idx (zone_id, date)
        ) ENGINE=InnoDB $charset_collate;";
        
        // Top Browsers (30 days)
        $table_browsers = $wpdb->prefix . 'cf_analytics_top_browsers';
        $sql_browsers = "CREATE TABLE $table_browsers (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            browser_name varchar(100) NOT NULL,
            page_views int(10) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY zone_date_browser (zone_id, date, browser_name),
            KEY date_idx (date),
            KEY zone_date_idx (zone_id, date)
        ) ENGINE=InnoDB $charset_collate;";
        
        // HTTP Status Codes (30 days)
        $table_status = $wpdb->prefix . 'cf_analytics_status_codes';
        $sql_status = "CREATE TABLE $table_status (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            status_code int(11) NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY zone_date_status (zone_id, date, status_code),
            KEY date_idx (date),
            KEY zone_date_idx (zone_id, date)
        ) ENGINE=InnoDB $charset_collate;";
        
        // Device Types (30 days)
        $table_devices = $wpdb->prefix . 'cf_analytics_device_types';
        $sql_devices = "CREATE TABLE $table_devices (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            device_type varchar(20) NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY zone_date_device (zone_id, date, device_type),
            KEY date_idx (date),
            KEY zone_date_idx (zone_id, date)
        ) ENGINE=InnoDB $charset_collate;";
        
        // User Agents (30 days, top 50 per day)
        $table_agents = $wpdb->prefix . 'cf_analytics_user_agents';
        $sql_agents = "CREATE TABLE $table_agents (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            zone_id varchar(32) NOT NULL,
            date date NOT NULL,
            user_agent varchar(500) NOT NULL,
            requests int(10) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY zone_date_ua (zone_id, date, user_agent(255)),
            KEY date_idx (date),
            KEY zone_date_idx (zone_id, date)
        ) ENGINE=InnoDB $charset_collate;";
        
        dbDelta($sql_hourly);
        dbDelta($sql_daily);
        dbDelta($sql_urls);
        dbDelta($sql_countries);
        dbDelta($sql_browsers);
        dbDelta($sql_status);
        dbDelta($sql_devices);
        dbDelta($sql_agents);
    }
    
    /**
     * Insert or update hourly data
     */
    public static function insert_hourly($zone_id, $datetime, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_hourly';
        
        $wpdb->replace($table, array_merge(
            array('zone_id' => $zone_id, 'datetime' => $datetime),
            $data
        ));
    }
    
    /**
     * Insert or update daily data
     */
    public static function insert_daily($zone_id, $date, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_daily';
        
        $wpdb->replace($table, array_merge(
            array('zone_id' => $zone_id, 'date' => $date),
            $data
        ));
    }
    
    /**
     * Insert top URL data
     */
    public static function insert_top_url($zone_id, $date, $url_path, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_urls';
        
        $wpdb->replace($table, array_merge(
            array(
                'zone_id' => $zone_id,
                'date' => $date,
                'url_path' => $url_path
            ),
            $data
        ));
    }
    
    /**
     * Insert country data
     */
    public static function insert_country($zone_id, $date, $country_code, $country_name, $data) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_countries';
        
        $wpdb->replace($table, array_merge(
            array(
                'zone_id' => $zone_id,
                'date' => $date,
                'country_code' => $country_code,
                'country_name' => $country_name
            ),
            $data
        ));
    }
    
    /**
     * Insert browser data
     */
    public static function insert_browser($zone_id, $date, $browser_name, $page_views) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_browsers';
        
        $wpdb->replace($table, array(
            'zone_id' => $zone_id,
            'date' => $date,
            'browser_name' => $browser_name,
            'page_views' => $page_views
        ));
    }
    
    /**
     * Insert HTTP status code data
     */
    public static function insert_status_code($zone_id, $date, $status_code, $requests) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_status_codes';
        
        $wpdb->replace($table, array(
            'zone_id' => $zone_id,
            'date' => $date,
            'status_code' => $status_code,
            'requests' => $requests
        ));
    }
    
    /**
     * Insert device type data
     */
    public static function insert_device_type($zone_id, $date, $device_type, $requests) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_device_types';
        
        $wpdb->replace($table, array(
            'zone_id' => $zone_id,
            'date' => $date,
            'device_type' => $device_type,
            'requests' => $requests
        ));
    }
    
    /**
     * Insert user agent data
     */
    public static function insert_user_agent($zone_id, $date, $user_agent, $requests) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_user_agents';
        
        $wpdb->replace($table, array(
            'zone_id' => $zone_id,
            'date' => $date,
            'user_agent' => $user_agent,
            'requests' => $requests
        ));
    }
    
    /**
     * Get hourly data for date range
     */
    public static function get_hourly_data($zone_id, $start_date, $end_date) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_hourly';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table 
            WHERE zone_id = %s 
            AND datetime >= %s 
            AND datetime <= %s 
            ORDER BY datetime ASC",
            $zone_id,
            $start_date,
            $end_date
        ));
    }
    
    /**
     * Get daily data for date range
     */
    public static function get_daily_data($zone_id, $start_date, $end_date) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_daily';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table 
            WHERE zone_id = %s 
            AND date >= %s 
            AND date <= %s 
            ORDER BY date ASC",
            $zone_id,
            $start_date,
            $end_date
        ));
    }
    
    /**
     * Get period totals
     */
    public static function get_period_totals($zone_id, $start_date, $end_date, $use_daily = false) {
        global $wpdb;
        
        $table = $use_daily 
            ? $wpdb->prefix . 'cf_analytics_daily'
            : $wpdb->prefix . 'cf_analytics_hourly';
        
        $date_field = $use_daily ? 'date' : 'datetime';
        
        $result = $wpdb->get_row($wpdb->prepare(
            "SELECT 
                SUM(requests) as total_requests,
                SUM(unique_visitors) as total_visitors,
                SUM(page_views) as total_pageviews,
                SUM(bandwidth_bytes) as total_bandwidth,
                SUM(cached_requests) as total_cached_requests,
                SUM(cached_bytes) as total_cached_bytes,
                SUM(threats) as total_threats
            FROM $table 
            WHERE zone_id = %s 
            AND $date_field >= %s 
            AND $date_field <= %s",
            $zone_id,
            $start_date,
            $end_date
        ));
        
        // Calculate cache hit rate
        if ($result && $result->total_requests > 0) {
            $result->cache_hit_rate = ($result->total_cached_requests / $result->total_requests) * 100;
        } else {
            $result->cache_hit_rate = 0;
        }
        
        return $result;
    }
    
    /**
     * Get top URLs for date range (filtered to exclude static assets)
     */
    public static function get_top_urls($zone_id, $start_date, $end_date, $limit = 20) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_urls';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT url_path, 
                    SUM(requests) as total_requests,
                    SUM(visits) as total_visits,
                    SUM(bandwidth_bytes) as total_bandwidth
            FROM $table 
            WHERE zone_id = %s 
            AND date >= %s 
            AND date <= %s
            GROUP BY url_path
            ORDER BY total_requests DESC
            LIMIT %d",
            $zone_id,
            $start_date,
            $end_date,
            $limit
        ));
    }
    
    /**
     * Get top content URLs (WordPress posts/pages only)
     */
    public static function get_top_content_urls($zone_id, $start_date, $end_date, $limit = 20) {
        // Get all URLs
        $all_urls = self::get_top_urls($zone_id, $start_date, $end_date, 999);
        
        // Get all post/page slugs
        $content_slugs = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        ));
        
        $valid_paths = array('/'); // Homepage is always valid
        
        foreach ($content_slugs as $post_id) {
            $permalink = get_permalink($post_id);
            $path = parse_url($permalink, PHP_URL_PATH);
            if ($path) {
                $valid_paths[] = rtrim($path, '/');
            }
        }
        
        // Filter URLs to only include WordPress content
        $content_urls = array_filter($all_urls, function($url) use ($valid_paths) {
            $clean_path = rtrim($url->url_path, '/');
            // Exclude any URLs with double slashes (malformed/bot requests)
            if (strpos($url->url_path, '//') !== false) {
                return false;
            }
            return in_array($clean_path, $valid_paths);
        });
        
        // Re-index array and limit
        $content_urls = array_values($content_urls);
        return array_slice($content_urls, 0, $limit);
    }
    
    /**
     * Get top countries for date range
     */
    public static function get_top_countries($zone_id, $start_date, $end_date, $limit = 20) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_countries';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT country_code,
                    country_name,
                    SUM(requests) as total_requests,
                    SUM(bandwidth_bytes) as total_bandwidth,
                    SUM(threats) as total_threats
            FROM $table 
            WHERE zone_id = %s 
            AND date >= %s 
            AND date <= %s 
            GROUP BY country_code, country_name
            ORDER BY total_requests DESC
            LIMIT %d",
            $zone_id,
            $start_date,
            $end_date,
            $limit
        ));
    }
    
    /**
     * Get top browsers for date range
     */
    public static function get_top_browsers($zone_id, $start_date, $end_date, $limit = 10) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_browsers';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT browser_name,
                    SUM(page_views) as total_pageviews
            FROM $table 
            WHERE zone_id = %s 
            AND date >= %s 
            AND date <= %s 
            GROUP BY browser_name
            ORDER BY total_pageviews DESC
            LIMIT %d",
            $zone_id,
            $start_date,
            $end_date,
            $limit
        ));
    }
    
    /**
     * Cleanup old hourly data (keep 7 days)
     */
    public static function cleanup_old_hourly_data() {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_hourly';
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE datetime < DATE_SUB(NOW(), INTERVAL 7 DAY)"
        ));
    }
    
    /**
     * Cleanup old URL data (keep 30 days)
     */
    public static function cleanup_old_url_data() {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_urls';
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE date < DATE_SUB(CURDATE(), INTERVAL 30 DAY)"
        ));
    }
    
    /**
     * Cleanup old country data (keep 30 days)
     */
    public static function cleanup_old_country_data() {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_countries';
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE date < DATE_SUB(CURDATE(), INTERVAL 30 DAY)"
        ));
    }
    
    /**
     * Cleanup old browser data (keep 30 days)
     */
    public static function cleanup_old_browser_data() {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_top_browsers';
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table WHERE date < DATE_SUB(CURDATE(), INTERVAL 30 DAY)"
        ));
    }
    
    /**
     * Optimize all tables
     */
    public static function optimize_tables() {
        global $wpdb;
        $prefix = $wpdb->prefix;
        
        $tables = array(
            $prefix . 'cf_analytics_hourly',
            $prefix . 'cf_analytics_daily',
            $prefix . 'cf_analytics_top_urls',
            $prefix . 'cf_analytics_top_countries',
            $prefix . 'cf_analytics_top_browsers'
        );
        
        foreach ($tables as $table) {
            $wpdb->query("OPTIMIZE TABLE $table");
        }
    }
    
    /**
     * Get database size statistics
     */
    public static function get_database_size() {
        global $wpdb;
        $prefix = $wpdb->prefix;
        
        $tables = array(
            'cf_analytics_hourly',
            'cf_analytics_daily',
            'cf_analytics_top_urls',
            'cf_analytics_top_countries',
            'cf_analytics_top_browsers'
        );
        
        $stats = array();
        $total_size = 0;
        $total_rows = 0;
        
        foreach ($tables as $table) {
            $full_table_name = $prefix . $table;
            $result = $wpdb->get_row($wpdb->prepare(
                "SELECT 
                    TABLE_ROWS as row_count,
                    DATA_LENGTH + INDEX_LENGTH as size_bytes
                FROM information_schema.TABLES 
                WHERE TABLE_SCHEMA = %s 
                AND TABLE_NAME = %s",
                DB_NAME,
                $full_table_name
            ));
            
            if ($result) {
                $stats[$table] = array(
                    'rows' => $result->row_count,
                    'size' => $result->size_bytes
                );
                $total_size += $result->size_bytes;
                $total_rows += $result->row_count;
            }
        }
        
        $stats['total'] = array(
            'rows' => $total_rows,
            'size' => $total_size
        );
        
        return $stats;
    }
    
    /**
     * Get HTTP status codes for date range
     */
    public static function get_status_codes($zone_id, $start_date, $end_date) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_status_codes';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT status_code, SUM(requests) as total_requests
            FROM $table
            WHERE zone_id = %s
            AND date >= %s
            AND date <= %s
            GROUP BY status_code
            ORDER BY total_requests DESC",
            $zone_id,
            $start_date,
            $end_date
        ));
    }
    
    /**
     * Get device types for date range
     */
    public static function get_device_types($zone_id, $start_date, $end_date) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_device_types';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT device_type, SUM(requests) as total_requests
            FROM $table
            WHERE zone_id = %s
            AND date >= %s
            AND date <= %s
            GROUP BY device_type
            ORDER BY total_requests DESC",
            $zone_id,
            $start_date,
            $end_date
        ));
    }
    
    /**
     * Get user agents for date range
     */
    public static function get_user_agents($zone_id, $start_date, $end_date, $limit = 50) {
        global $wpdb;
        $table = $wpdb->prefix . 'cf_analytics_user_agents';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT user_agent, SUM(requests) as total_requests
            FROM $table
            WHERE zone_id = %s
            AND date >= %s
            AND date <= %s
            GROUP BY user_agent
            ORDER BY total_requests DESC
            LIMIT %d",
            $zone_id,
            $start_date,
            $end_date,
            $limit
        ));
    }
}
