<?php
/**
 * Cron job scheduler and handlers
 * 
 * Jobs:
 * - hourly_sync: Pull time-series metrics every hour
 * - daily_sync: Pull top URLs, countries, browsers once per day
 * - daily_cleanup: Clean old data and aggregate to daily table
 * - monthly_optimize: Optimize database tables
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF_Cron {
    
    /**
     * Schedule all cron events
     */
    public static function schedule_events() {
        if (!wp_next_scheduled('cf_analytics_hourly_sync')) {
            wp_schedule_event(time(), 'hourly', 'cf_analytics_hourly_sync');
        }
        
        if (!wp_next_scheduled('cf_analytics_hourly_detailed_sync')) {
            wp_schedule_event(time(), 'hourly', 'cf_analytics_hourly_detailed_sync');
        }
        
        if (!wp_next_scheduled('cf_analytics_daily_sync')) {
            wp_schedule_event(strtotime('03:00:00'), 'daily', 'cf_analytics_daily_sync');
        }
        
        if (!wp_next_scheduled('cf_analytics_daily_cleanup')) {
            wp_schedule_event(strtotime('02:00:00'), 'daily', 'cf_analytics_daily_cleanup');
        }
        
        if (!wp_next_scheduled('cf_analytics_monthly_optimize')) {
            wp_schedule_event(strtotime('first day of next month 03:00:00'), 'monthly', 'cf_analytics_monthly_optimize');
        }
    }
    
    /**
     * Clear all scheduled events
     */
    public static function clear_scheduled_events() {
        wp_clear_scheduled_hook('cf_analytics_hourly_sync');
        wp_clear_scheduled_hook('cf_analytics_hourly_detailed_sync');
        wp_clear_scheduled_hook('cf_analytics_daily_sync');
        wp_clear_scheduled_hook('cf_analytics_daily_cleanup');
        wp_clear_scheduled_hook('cf_analytics_monthly_optimize');
    }
    
    /**
     * Hourly sync - pull time-series metrics
     * Pulls last 2 hours of data (overlap for safety)
     */
    public static function hourly_sync() {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['zone_id'])) {
            return;
        }
        
        $api = new CF_API();
        $zone_id = $settings['zone_id'];
        
        // Fetch last 2 hours
        $end_time = gmdate('Y-m-d\TH:i:s\Z');
        $start_time = gmdate('Y-m-d\TH:i:s\Z', strtotime('-2 hours'));
        
        $result = $api->get_hourly_analytics($zone_id, $start_time, $end_time);
        
        if (!$result['success'] || empty($result['data'])) {
            return;
        }
        
        foreach ($result['data'] as $row) {
            $data = array(
                'requests' => $row['sum']['requests'] ?? 0,
                'unique_visitors' => $row['uniq']['uniques'] ?? 0,
                'page_views' => $row['sum']['pageViews'] ?? 0,
                'bandwidth_bytes' => $row['sum']['bytes'] ?? 0,
                'cached_requests' => $row['sum']['cachedRequests'] ?? 0,
                'cached_bytes' => $row['sum']['cachedBytes'] ?? 0,
                'threats' => $row['sum']['threats'] ?? 0
            );
            
            // Convert ISO 8601 to MySQL datetime
            $datetime_iso = $row['dimensions']['datetime'];
            $datetime_mysql = gmdate('Y-m-d H:i:s', strtotime($datetime_iso));
            
            CF_Database::insert_hourly($zone_id, $datetime_mysql, $data);
        }
        
        update_option('cf_analytics_last_sync', current_time('mysql'));
    }
    
    /**
     * Daily sync - pull top URLs, countries, browsers
     * Runs once per day for yesterday's data
     */
    public static function daily_sync() {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['zone_id'])) {
            return;
        }
        
        $api = new CF_API();
        $zone_id = $settings['zone_id'];
        
        // Get yesterday's date
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        $start_time = gmdate('Y-m-d\T00:00:00\Z', strtotime($yesterday));
        $end_time = gmdate('Y-m-d\T23:59:59\Z', strtotime($yesterday));
        
        self::sync_detailed_data($api, $zone_id, $yesterday, $start_time, $end_time);
    }
    
    /**
     * Hourly detailed sync - pull today's data every hour
     * Keeps "Today" view up-to-date throughout the day
     */
    public static function hourly_detailed_sync() {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['zone_id'])) {
            return;
        }
        
        $api = new CF_API();
        $zone_id = $settings['zone_id'];
        
        // Get today's date
        $today = date('Y-m-d');
        $start_time = gmdate('Y-m-d\T00:00:00\Z', strtotime($today));
        $end_time = gmdate('Y-m-d\TH:i:s\Z'); // Current time
        
        self::sync_detailed_data($api, $zone_id, $today, $start_time, $end_time);
    }
    
    /**
     * Sync detailed data (URLs, countries, browsers, etc.) for a specific date
     * Used by both daily_sync and hourly_detailed_sync
     */
    private static function sync_detailed_data($api, $zone_id, $date, $start_time, $end_time) {
        // Fetch top URLs
        $urls = $api->get_top_urls($zone_id, $start_time, $end_time, 50);
        if ($urls['success'] && !empty($urls['data'])) {
            foreach ($urls['data'] as $row) {
                $url_path = $row['dimensions']['clientRequestPath'] ?? '/';
                CF_Database::insert_top_url($zone_id, $date, $url_path, array(
                    'requests' => $row['count'] ?? 0,
                    'visits' => $row['sum']['visits'] ?? 0,
                    'bandwidth_bytes' => $row['sum']['edgeResponseBytes'] ?? 0
                ));
            }
        }
        
        // Fetch country data
        $countries = $api->get_country_data($zone_id, $start_time, $end_time, 50);
        if ($countries['success'] && !empty($countries['data'])) {
            foreach ($countries['data'] as $row) {
                $country_code = $row['dimensions']['clientCountryName'] ?? 'XX';
                $country_name = CF_API::get_country_name($country_code);
                
                CF_Database::insert_country($zone_id, $date, $country_code, $country_name, array(
                    'requests' => $row['count'] ?? 0,
                    'bandwidth_bytes' => 0,
                    'threats' => 0
                ));
            }
        }
        
        // Fetch browser data
        $browsers = $api->get_browser_data($zone_id, $start_time, $end_time, 20);
        if ($browsers['success'] && !empty($browsers['data'])) {
            foreach ($browsers['data'] as $row) {
                $browser_name = $row['dimensions']['uaBrowserFamily'] ?? 'Unknown';
                CF_Database::insert_browser($zone_id, $date, $browser_name, $row['count'] ?? 0);
            }
        }
        
        // Fetch HTTP status codes
        $status_codes = $api->get_http_status_codes($zone_id, $start_time, $end_time, 50);
        if ($status_codes['success'] && !empty($status_codes['data'])) {
            foreach ($status_codes['data'] as $row) {
                $status_code = $row['dimensions']['edgeResponseStatus'] ?? 0;
                CF_Database::insert_status_code($zone_id, $date, $status_code, $row['count'] ?? 0);
            }
        }
        
        // Fetch device types
        $device_types = $api->get_device_types($zone_id, $start_time, $end_time);
        if ($device_types['success'] && !empty($device_types['data'])) {
            foreach ($device_types['data'] as $row) {
                $device_type = $row['dimensions']['clientDeviceType'] ?? 'unknown';
                CF_Database::insert_device_type($zone_id, $date, $device_type, $row['count'] ?? 0);
            }
        }
        
        // Fetch top user agents
        $user_agents = $api->get_user_agents($zone_id, $start_time, $end_time, 50);
        if ($user_agents['success'] && !empty($user_agents['data'])) {
            foreach ($user_agents['data'] as $row) {
                $user_agent = $row['dimensions']['userAgent'] ?? 'Unknown';
                CF_Database::insert_user_agent($zone_id, $date, $user_agent, $row['count'] ?? 0);
            }
        }
    }
    
    /**
     * Daily cleanup
     * - Aggregate yesterday's hourly data to daily table
     * - Delete hourly data older than 7 days
     * - Delete URL/country/browser data older than 30 days
     */
    public static function daily_cleanup() {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['zone_id'])) {
            return;
        }
        
        $zone_id = $settings['zone_id'];
        
        // Aggregate yesterday's hourly data to daily table
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        $start = $yesterday . ' 00:00:00';
        $end = $yesterday . ' 23:59:59';
        
        $totals = CF_Database::get_period_totals($zone_id, $start, $end, false);
        
        if ($totals && $totals->total_requests > 0) {
            CF_Database::insert_daily($zone_id, $yesterday, array(
                'requests' => $totals->total_requests,
                'unique_visitors' => $totals->total_visitors,
                'page_views' => $totals->total_pageviews,
                'bandwidth_bytes' => $totals->total_bandwidth,
                'cached_requests' => $totals->total_cached_requests,
                'cached_bytes' => $totals->total_cached_bytes,
                'threats' => $totals->total_threats
            ));
        }
        
        // Clean up old data
        CF_Database::cleanup_old_hourly_data();
        CF_Database::cleanup_old_url_data();
        CF_Database::cleanup_old_country_data();
        CF_Database::cleanup_old_browser_data();
        
        // Note: Status codes, device types, and user agents now have unlimited retention
        // for historical analysis (Last Year, All Time views)
    }
    
    /**
     * Monthly optimization
     * Optimize all database tables
     */
    public static function monthly_optimize() {
        CF_Database::optimize_tables();
    }
    
    /**
     * Backfill historical data
     * Pull X days of historical data (useful on initial setup)
     */
    public static function backfill_data($days = 7) {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['zone_id'])) {
            return array('success' => false, 'message' => 'No zone configured');
        }
        
        $api = new CF_API();
        $zone_id = $settings['zone_id'];
        
        $days = min($days, 30); // Max 30 days
        $total_inserted = 0;
        
        // Pull hourly data for each day
        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', strtotime("-$i days"));
            $start_time = gmdate('Y-m-d\T00:00:00\Z', strtotime($date));
            $end_time = gmdate('Y-m-d\T23:59:59\Z', strtotime($date));
            
            $result = $api->get_hourly_analytics($zone_id, $start_time, $end_time);
            
            if ($result['success'] && !empty($result['data'])) {
                foreach ($result['data'] as $row) {
                    $data = array(
                        'requests' => $row['sum']['requests'] ?? 0,
                        'unique_visitors' => $row['uniq']['uniques'] ?? 0,
                        'page_views' => $row['sum']['pageViews'] ?? 0,
                        'bandwidth_bytes' => $row['sum']['bytes'] ?? 0,
                        'cached_requests' => $row['sum']['cachedRequests'] ?? 0,
                        'cached_bytes' => $row['sum']['cachedBytes'] ?? 0,
                        'threats' => $row['sum']['threats'] ?? 0
                    );
                    
                    $datetime_iso = $row['dimensions']['datetime'];
                    $datetime_mysql = gmdate('Y-m-d H:i:s', strtotime($datetime_iso));
                    
                    CF_Database::insert_hourly($zone_id, $datetime_mysql, $data);
                    $total_inserted++;
                }
            }
            
            // Also pull detailed data for this day
            $urls = $api->get_top_urls($zone_id, $start_time, $end_time, 50);
            if ($urls['success'] && !empty($urls['data'])) {
                foreach ($urls['data'] as $row) {
                    $url_path = $row['dimensions']['clientRequestPath'] ?? '/';
                    CF_Database::insert_top_url($zone_id, $date, $url_path, array(
                        'requests' => $row['count'] ?? 0,
                        'visits' => $row['sum']['visits'] ?? 0,
                        'bandwidth_bytes' => $row['sum']['edgeResponseBytes'] ?? 0
                    ));
                }
            }
            
            $countries = $api->get_country_data($zone_id, $start_time, $end_time, 50);
            if ($countries['success'] && !empty($countries['data'])) {
                foreach ($countries['data'] as $row) {
                    $country_code = $row['dimensions']['clientCountryName'] ?? 'XX'; // Already a code
                    $country_name = CF_API::get_country_name($country_code);
                    
                    CF_Database::insert_country($zone_id, $date, $country_code, $country_name, array(
                        'requests' => $row['count'] ?? 0,
                        'bandwidth_bytes' => 0, // Not available on free plan
                        'threats' => 0 // Not available on free plan
                    ));
                }
            }
            
            $browsers = $api->get_browser_data($zone_id, $start_time, $end_time, 20);
            if ($browsers['success'] && !empty($browsers['data'])) {
                foreach ($browsers['data'] as $row) {
                    $browser_name = $row['dimensions']['uaBrowserFamily'] ?? 'Unknown';
                    CF_Database::insert_browser($zone_id, $date, $browser_name, $row['count'] ?? 0);
                }
            }
            
            // Status codes
            $status_codes = $api->get_http_status_codes($zone_id, $start_time, $end_time, 50);
            if ($status_codes['success'] && !empty($status_codes['data'])) {
                foreach ($status_codes['data'] as $row) {
                    $status_code = $row['dimensions']['edgeResponseStatus'] ?? 0;
                    CF_Database::insert_status_code($zone_id, $date, $status_code, $row['count'] ?? 0);
                }
            }
            
            // Device types
            $device_types = $api->get_device_types($zone_id, $start_time, $end_time);
            if ($device_types['success'] && !empty($device_types['data'])) {
                foreach ($device_types['data'] as $row) {
                    $device_type = $row['dimensions']['clientDeviceType'] ?? 'unknown';
                    CF_Database::insert_device_type($zone_id, $date, $device_type, $row['count'] ?? 0);
                }
            }
            
            // User agents
            $user_agents = $api->get_user_agents($zone_id, $start_time, $end_time, 50);
            if ($user_agents['success'] && !empty($user_agents['data'])) {
                foreach ($user_agents['data'] as $row) {
                    $user_agent = $row['dimensions']['userAgent'] ?? 'Unknown';
                    CF_Database::insert_user_agent($zone_id, $date, $user_agent, $row['count'] ?? 0);
                }
            }
            
            // Small delay to avoid rate limits
            usleep(200000); // 200ms
        }
        
        return array(
            'success' => true,
            'message' => "Backfilled $total_inserted hourly records for $days days",
            'days' => $days,
            'records' => $total_inserted
        );
    }
}

// Register cron hooks
add_action('cf_analytics_hourly_sync', array('CF_Cron', 'hourly_sync'));
add_action('cf_analytics_hourly_detailed_sync', array('CF_Cron', 'hourly_detailed_sync'));
add_action('cf_analytics_daily_sync', array('CF_Cron', 'daily_sync'));
add_action('cf_analytics_daily_cleanup', array('CF_Cron', 'daily_cleanup'));
add_action('cf_analytics_monthly_optimize', array('CF_Cron', 'monthly_optimize'));
