<?php
/**
 * Cloudflare API interactions
 * 
 * Handles:
 * - REST API (zones list, credentials test)
 * - GraphQL API (analytics queries)
 *   - httpRequests1hGroups: Time-series metrics
 *   - httpRequestsAdaptiveGroups: Top URLs, countries, browsers
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF_API {
    private $api_url = 'https://api.cloudflare.com/client/v4';
    private $graphql_url = 'https://api.cloudflare.com/client/v4/graphql';
    
    /**
     * Get authentication headers
     */
    private function get_headers() {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['email']) || empty($settings['api_key'])) {
            return array();
        }
        
        $email = CF_Encryption::decrypt($settings['email']);
        $api_key = CF_Encryption::decrypt($settings['api_key']);
        
        return array(
            'X-Auth-Email' => $email,
            'X-Auth-Key' => $api_key,
            'Content-Type' => 'application/json'
        );
    }
    
    /**
     * Test API credentials
     */
    public function test_credentials($email, $api_key) {
        $response = wp_remote_get($this->api_url . '/user', array(
            'headers' => array(
                'X-Auth-Email' => $email,
                'X-Auth-Key' => $api_key
            )
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['success']) && $body['success']) {
            return array('success' => true, 'data' => $body['result']);
        }
        
        return array('success' => false, 'message' => $body['errors'][0]['message'] ?? 'Authentication failed');
    }
    
    /**
     * Get list of zones
     */
    public function get_zones() {
        $response = wp_remote_get($this->api_url . '/zones', array(
            'headers' => $this->get_headers()
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['success']) && $body['success']) {
            return array('success' => true, 'data' => $body['result']);
        }
        
        return array('success' => false, 'message' => 'Failed to fetch zones');
    }
    
    /**
     * Get hourly time-series analytics
     * Returns: requests, visitors, bandwidth, cache stats, threats
     */
    public function get_hourly_analytics($zone_id, $start_datetime, $end_datetime) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequests1hGroups(
                        limit: 10000,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        }
                    ) {
                        dimensions {
                            datetime
                        }
                        sum {
                            requests
                            pageViews
                            bytes
                            cachedRequests
                            cachedBytes
                            threats
                        }
                        uniq {
                            uniques
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequests1hGroups');
    }
    
    /**
     * Get top URLs for a date range
     * Uses httpRequestsAdaptiveGroups with clientRequestPath dimension
     */
    public function get_top_urls($zone_id, $start_datetime, $end_datetime, $limit = 50) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientRequestPath
                        }
                        sum {
                            visits
                            edgeResponseBytes
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get geographic data (countries)
     * Uses httpRequestsAdaptiveGroups with clientCountryName dimension
     */
    public function get_country_data($zone_id, $start_datetime, $end_datetime, $limit = 50) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientCountryName
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get browser statistics
     * Uses httpRequestsAdaptiveGroups with uaBrowserFamily dimension
     */
    public function get_browser_data($zone_id, $start_datetime, $end_datetime, $limit = 20) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [sum_visits_DESC]
                    ) {
                        dimensions {
                            uaBrowserFamily
                        }
                        sum {
                            visits
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Execute GraphQL query
     */
    private function execute_graphql_query($query, $variables, $dataset_name) {
        $response = wp_remote_post($this->graphql_url, array(
            'headers' => $this->get_headers(),
            'body' => json_encode(array(
                'query' => $query,
                'variables' => $variables
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array('success' => false, 'message' => $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        // Check for successful data
        if (isset($body['data']['viewer']['zones'][0][$dataset_name])) {
            return array(
                'success' => true,
                'data' => $body['data']['viewer']['zones'][0][$dataset_name]
            );
        }
        
        // Handle errors
        if (isset($body['errors'])) {
            return array(
                'success' => false,
                'message' => 'GraphQL query failed',
                'errors' => $body['errors']
            );
        }
        
        return array('success' => false, 'message' => 'No data returned', 'body' => $body);
    }
    
    /**
     * Format number for display
     */
    public static function format_number($number) {
        if ($number >= 1000000) {
            return round($number / 1000000, 1) . 'M';
        } elseif ($number >= 1000) {
            return round($number / 1000, 1) . 'K';
        }
        return number_format($number);
    }
    
    /**
     * Format bytes to human readable (always show KB minimum)
     */
    public static function format_bytes($bytes, $precision = 2) {
        $units = array('KB', 'MB', 'GB', 'TB');
        
        $bytes = max($bytes, 0);
        
        if ($bytes == 0) {
            return '0 KB';
        }
        
        // Convert to KB first
        $bytes = $bytes / 1024;
        
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, $precision) . ' ' . $units[$pow];
    }
    
    /**
     * Get country code from country name
     * Simple mapping for common countries
     */
    /**
     * Get country name from 2-letter ISO code
     */
    public static function get_country_name($code) {
        $countries = array(
            'US' => 'United States', 'GB' => 'United Kingdom', 'CA' => 'Canada', 'AU' => 'Australia',
            'DE' => 'Germany', 'FR' => 'France', 'IT' => 'Italy', 'ES' => 'Spain', 'NL' => 'Netherlands',
            'BE' => 'Belgium', 'CH' => 'Switzerland', 'AT' => 'Austria', 'SE' => 'Sweden', 'NO' => 'Norway',
            'DK' => 'Denmark', 'FI' => 'Finland', 'PL' => 'Poland', 'CZ' => 'Czech Republic', 'HU' => 'Hungary',
            'RO' => 'Romania', 'BG' => 'Bulgaria', 'GR' => 'Greece', 'PT' => 'Portugal', 'IE' => 'Ireland',
            'JP' => 'Japan', 'CN' => 'China', 'IN' => 'India', 'KR' => 'South Korea', 'TH' => 'Thailand',
            'VN' => 'Vietnam', 'ID' => 'Indonesia', 'MY' => 'Malaysia', 'PH' => 'Philippines', 'SG' => 'Singapore',
            'BR' => 'Brazil', 'MX' => 'Mexico', 'AR' => 'Argentina', 'CL' => 'Chile', 'CO' => 'Colombia',
            'PE' => 'Peru', 'VE' => 'Venezuela', 'RU' => 'Russia', 'UA' => 'Ukraine', 'TR' => 'Turkey',
            'SA' => 'Saudi Arabia', 'AE' => 'United Arab Emirates', 'IL' => 'Israel', 'EG' => 'Egypt',
            'ZA' => 'South Africa', 'NG' => 'Nigeria', 'KE' => 'Kenya', 'NZ' => 'New Zealand',
            'UZ' => 'Uzbekistan', 'XX' => 'Unknown'
        );
        
        return $countries[$code] ?? $code;
    }
    
    /**
     * Get HTTP status codes breakdown
     */
    public function get_http_status_codes($zone_id, $start_datetime, $end_datetime, $limit = 20) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            edgeResponseStatus
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get device types breakdown
     */
    public function get_device_types($zone_id, $start_datetime, $end_datetime) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: 10,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientDeviceType
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get HTTP methods breakdown
     */
    public function get_http_methods($zone_id, $start_datetime, $end_datetime) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: 10,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientRequestHTTPMethodName
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get top user agents
     */
    public function get_user_agents($zone_id, $start_datetime, $end_datetime, $limit = 50) {
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            userAgent
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get daily URL data for a specific date
     * Uses httpRequestsAdaptiveGroups with date dimension for daily snapshots
     */
    public function get_daily_urls($zone_id, $date, $limit = 50) {
        $start_datetime = $date . 'T00:00:00Z';
        $end_datetime = date('Y-m-d', strtotime($date . ' +1 day')) . 'T00:00:00Z';
        
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientRequestPath
                        }
                        sum {
                            visits
                            edgeResponseBytes
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get daily country data for a specific date
     */
    public function get_daily_countries($zone_id, $date, $limit = 50) {
        $start_datetime = $date . 'T00:00:00Z';
        $end_datetime = date('Y-m-d', strtotime($date . ' +1 day')) . 'T00:00:00Z';
        
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientCountryName
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get daily browser data for a specific date
     */
    public function get_daily_browsers($zone_id, $date, $limit = 20) {
        $start_datetime = $date . 'T00:00:00Z';
        $end_datetime = date('Y-m-d', strtotime($date . ' +1 day')) . 'T00:00:00Z';
        
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [sum_visits_DESC]
                    ) {
                        dimensions {
                            uaBrowserFamily
                        }
                        sum {
                            visits
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get daily status codes for a specific date
     */
    public function get_daily_status_codes($zone_id, $date, $limit = 20) {
        $start_datetime = $date . 'T00:00:00Z';
        $end_datetime = date('Y-m-d', strtotime($date . ' +1 day')) . 'T00:00:00Z';
        
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            edgeResponseStatus
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get daily device types for a specific date
     */
    public function get_daily_device_types($zone_id, $date) {
        $start_datetime = $date . 'T00:00:00Z';
        $end_datetime = date('Y-m-d', strtotime($date . ' +1 day')) . 'T00:00:00Z';
        
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: 10,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            clientDeviceType
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
    
    /**
     * Get daily user agents for a specific date
     */
    public function get_daily_user_agents($zone_id, $date, $limit = 50) {
        $start_datetime = $date . 'T00:00:00Z';
        $end_datetime = date('Y-m-d', strtotime($date . ' +1 day')) . 'T00:00:00Z';
        
        $query = '
        query ($zoneTag: string, $start: Time, $end: Time, $limit: int) {
            viewer {
                zones(filter: {zoneTag: $zoneTag}) {
                    httpRequestsAdaptiveGroups(
                        limit: $limit,
                        filter: {
                            datetime_geq: $start,
                            datetime_lt: $end
                        },
                        orderBy: [count_DESC]
                    ) {
                        count
                        dimensions {
                            userAgent
                        }
                    }
                }
            }
        }';
        
        $variables = array(
            'zoneTag' => $zone_id,
            'start' => $start_datetime,
            'end' => $end_datetime,
            'limit' => $limit
        );
        
        return $this->execute_graphql_query($query, $variables, 'httpRequestsAdaptiveGroups');
    }
}
