<?php
/**
 * WordPress Dashboard Widget
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF_Analytics_Widget {
    
    public function __construct() {
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
    }
    
    /**
     * Register dashboard widget
     */
    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'cf_analytics_widget',
            '<span class="dashicons dashicons-cloud" style="color: #f38020;"></span> Cloudflare Analytics',
            array($this, 'render_widget')
        );
    }
    
    /**
     * Calculate percentage change
     */
    private function calculate_change($current, $previous) {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        return (($current - $previous) / $previous) * 100;
    }
    
    /**
     * Render change badge
     */
    private function render_change_badge($change) {
        if ($change > 0) {
            $class = 'positive';
            $icon = '↑';
        } elseif ($change < 0) {
            $class = 'negative';
            $icon = '↓';
        } else {
            $class = 'neutral';
            $icon = '→';
        }
        
        return sprintf(
            '<span class="cf-widget-change %s">%s %.1f%%</span>',
            $class,
            $icon,
            abs($change)
        );
    }
    
    /**
     * Render widget content
     */
    public function render_widget() {
        $settings = get_option('cf_analytics_settings', array());
        
        if (empty($settings['zone_id'])) {
            ?>
            <div style="padding: 20px; text-align: center; background: #fff3cd; border-radius: 8px; border-left: 4px solid #ffc107;">
                <p style="margin: 0;">⚠️ Please <a href="<?php echo admin_url('admin.php?page=cf-analytics-settings'); ?>" style="font-weight: 600;">configure Cloudflare Analytics</a> to see your stats.</p>
            </div>
            <?php
            return;
        }
        
        $zone_id = $settings['zone_id'];
        $period = isset($_GET['cf_period']) ? $_GET['cf_period'] : 'today';
        
        // Define periods
        $periods = array(
            'today' => array('label' => 'Today', 'days' => 0),
            '7days' => array('label' => '7 Days', 'days' => 7),
            '30days' => array('label' => '30 Days', 'days' => 30)
        );
        
        // Calculate dates (use UTC since CF stores in UTC)
        if ($period === 'today') {
            // Get today in UTC
            $start_date = gmdate('Y-m-d') . ' 00:00:00';
            $end_date = gmdate('Y-m-d H:i:s');
        } else {
            $days = $periods[$period]['days'];
            $start_date = gmdate('Y-m-d H:i:s', strtotime("-$days days"));
            $end_date = gmdate('Y-m-d H:i:s');
        }
        
        // Get totals for current period
        $totals = CF_Database::get_period_totals($zone_id, $start_date, $end_date);
        
        // Get previous period for comparison
        $time_diff = strtotime($end_date) - strtotime($start_date);
        $prev_end = $start_date;
        $prev_start = gmdate('Y-m-d H:i:s', strtotime($prev_end) - $time_diff);
        $prev_totals = CF_Database::get_period_totals($zone_id, $prev_start, $prev_end);
        
        // Get top 5 content URLs
        $top_urls = CF_Database::get_top_content_urls($zone_id, gmdate('Y-m-d', strtotime($start_date)), gmdate('Y-m-d'), 5);
        
        // Calculate percentage changes
        $requests_change = $this->calculate_change($totals->total_requests ?? 0, $prev_totals->total_requests ?? 0);
        $visitors_change = $this->calculate_change($totals->total_visitors ?? 0, $prev_totals->total_visitors ?? 0);
        $bandwidth_change = $this->calculate_change($totals->total_bandwidth ?? 0, $prev_totals->total_bandwidth ?? 0);
        $cache_change = $this->calculate_change($totals->cache_hit_rate ?? 0, $prev_totals->cache_hit_rate ?? 0);
        
        ?>
        <div class="cf-widget-container">
            <!-- Period Selector -->
            <div class="cf-widget-period-selector">
                <?php foreach ($periods as $key => $p): ?>
                    <a href="<?php echo add_query_arg('cf_period', $key); ?>" 
                       class="cf-period-btn <?php echo $period === $key ? 'active' : ''; ?>">
                        <?php echo $p['label']; ?>
                    </a>
                <?php endforeach; ?>
            </div>
            
            <div class="cf-widget-grid">
                <div class="cf-widget-stat-item">
                    <div class="cf-widget-icon" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                        <span class="dashicons dashicons-chart-line"></span>
                    </div>
                    <div class="cf-widget-content">
                        <div class="cf-widget-label">Requests</div>
                        <div class="cf-widget-value"><?php echo CF_API::format_number($totals->total_requests ?? 0); ?></div>
                        <?php echo $this->render_change_badge($requests_change); ?>
                    </div>
                </div>
                
                <div class="cf-widget-stat-item">
                    <div class="cf-widget-icon" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                        <span class="dashicons dashicons-groups"></span>
                    </div>
                    <div class="cf-widget-content">
                        <div class="cf-widget-label">Visitors</div>
                        <div class="cf-widget-value"><?php echo CF_API::format_number($totals->total_visitors ?? 0); ?></div>
                        <?php echo $this->render_change_badge($visitors_change); ?>
                    </div>
                </div>
                
                <div class="cf-widget-stat-item">
                    <div class="cf-widget-icon" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
                        <span class="dashicons dashicons-download"></span>
                    </div>
                    <div class="cf-widget-content">
                        <div class="cf-widget-label">Bandwidth</div>
                        <div class="cf-widget-value"><?php echo CF_API::format_bytes($totals->total_bandwidth ?? 0); ?></div>
                        <?php echo $this->render_change_badge($bandwidth_change); ?>
                    </div>
                </div>
                
                <div class="cf-widget-stat-item">
                    <div class="cf-widget-icon" style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);">
                        <span class="dashicons dashicons-performance"></span>
                    </div>
                    <div class="cf-widget-content">
                        <div class="cf-widget-label">Cache Hit</div>
                        <div class="cf-widget-value"><?php echo round($totals->cache_hit_rate ?? 0, 1); ?>%</div>
                        <?php echo $this->render_change_badge($cache_change); ?>
                    </div>
                </div>
            </div>
            
            <?php if (!empty($top_urls)): ?>
            <div style="padding: 12px; border-top: 1px solid #e0e0e0;">
                <h4 style="margin: 0 0 10px 0; font-size: 12px; color: #666; text-transform: uppercase; letter-spacing: 0.3px;">Top Content</h4>
                <div style="display: flex; flex-direction: column; gap: 8px;">
                    <?php foreach ($top_urls as $url): ?>
                        <?php
                        // Get post title
                        $display_title = $url->url_path;
                        $post_id = url_to_postid($url->url_path);
                        if ($post_id) {
                            $display_title = get_the_title($post_id);
                        } elseif ($url->url_path === '/' || $url->url_path === '') {
                            $display_title = 'Homepage';
                        }
                        
                        if (strlen($display_title) > 40) {
                            $display_title = substr($display_title, 0, 37) . '...';
                        }
                        ?>
                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 8px; background: #f8f9fa; border-radius: 4px;">
                            <div style="flex: 1; min-width: 0; overflow: hidden;">
                                <div style="font-size: 12px; font-weight: 500; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;"><?php echo esc_html($display_title); ?></div>
                            </div>
                            <div style="margin-left: 10px; font-size: 12px; font-weight: 600; color: #667eea; flex-shrink: 0;">
                                <?php echo CF_API::format_number($url->total_requests); ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="cf-widget-footer">
                <a href="<?php echo admin_url('admin.php?page=cf-analytics'); ?>" class="button button-primary" style="width: 100%; text-align: center; justify-content: center;">
                    View Full Analytics Dashboard →
                </a>
            </div>
        </div>
        
        <style>
        .cf-widget-container {
            margin: -12px;
        }
        .cf-widget-period-selector {
            display: flex;
            gap: 8px;
            padding: 12px;
            padding-bottom: 0;
        }
        .cf-period-btn {
            flex: 1;
            padding: 6px 12px;
            text-align: center;
            background: #f0f0f1;
            border: 1px solid #dcdcde;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 500;
            color: #50575e;
            text-decoration: none;
            transition: all 0.2s;
        }
        .cf-period-btn:hover {
            background: #e8e8e9;
            color: #1d2327;
        }
        .cf-period-btn.active {
            background: #2271b1;
            color: #fff;
            border-color: #2271b1;
        }
        .cf-widget-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 12px;
            padding: 12px;
        }
        .cf-widget-stat-item {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px;
            background: #f8f9fa;
            border-radius: 8px;
            transition: all 0.2s;
        }
        .cf-widget-stat-item:hover {
            background: #f0f0f1;
            transform: translateY(-1px);
        }
        .cf-widget-icon {
            width: 44px;
            height: 44px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            flex-shrink: 0;
        }
        .cf-widget-icon .dashicons {
            font-size: 22px;
            width: 22px;
            height: 22px;
            color: #fff;
        }
        .cf-widget-content {
            flex: 1;
            min-width: 0;
        }
        .cf-widget-label {
            font-size: 11px;
            color: #6c757d;
            text-transform: uppercase;
            letter-spacing: 0.3px;
            margin-bottom: 2px;
            font-weight: 500;
        }
        .cf-widget-value {
            font-size: 20px;
            font-weight: 600;
            color: #1d2327;
            line-height: 1.2;
            margin-bottom: 3px;
        }
        .cf-widget-change {
            font-size: 11px;
            padding: 2px 6px;
            border-radius: 4px;
            display: inline-block;
            font-weight: 500;
        }
        .cf-widget-change.positive {
            background: #d4edda;
            color: #155724;
        }
        .cf-widget-change.negative {
            background: #f8d7da;
            color: #721c24;
        }
        .cf-widget-change.neutral {
            background: #e2e3e5;
            color: #383d41;
        }
        .cf-widget-footer {
            padding: 12px;
            padding-top: 0;
        }
        .cf-widget-footer .button {
            height: auto;
            padding: 10px 16px;
            font-size: 13px;
        }
        </style>
        <?php
    }
}

new CF_Analytics_Widget();
