<?php
/**
 * Settings page for Cloudflare Analytics
 */

if (!defined('ABSPATH')) {
    exit;
}

// Add menu items
add_action('admin_menu', function() {
    add_menu_page(
        'Cloudflare Analytics',
        'CF Analytics',
        'manage_options',
        'cf-analytics',
        'cf_analytics_page',
        'dashicons-cloud',
        30
    );
    
    add_submenu_page(
        'cf-analytics',
        'Analytics Dashboard',
        'Dashboard',
        'manage_options',
        'cf-analytics',
        'cf_analytics_page'
    );
    
    add_submenu_page(
        'cf-analytics',
        'Settings',
        'Settings',
        'manage_options',
        'cf-analytics-settings',
        'cf_analytics_settings_page'
    );
});

/**
 * Handle settings form submission
 */
add_action('admin_init', function() {
    if (isset($_POST['cf_analytics_save_settings'])) {
        check_admin_referer('cf_analytics_settings');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $email = sanitize_email($_POST['cf_email']);
        $api_key = sanitize_text_field($_POST['cf_api_key']);
        $zone_id = sanitize_text_field($_POST['cf_zone_id']);
        
        // Test credentials
        $api = new CF_API();
        $test = $api->test_credentials($email, $api_key);
        
        if (!$test['success']) {
            add_settings_error('cf_analytics', 'credentials_invalid', 'Invalid credentials: ' . $test['message']);
            return;
        }
        
        // Encrypt and save
        $settings = array(
            'email' => CF_Encryption::encrypt($email),
            'api_key' => CF_Encryption::encrypt($api_key),
            'zone_id' => $zone_id
        );
        
        update_option('cf_analytics_settings', $settings);
        add_settings_error('cf_analytics', 'settings_saved', 'Settings saved successfully!', 'success');
    }
    
    // Handle disconnect
    if (isset($_POST['cf_analytics_disconnect'])) {
        check_admin_referer('cf_analytics_disconnect');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        delete_option('cf_analytics_settings');
        add_settings_error('cf_analytics', 'disconnected', 'Disconnected successfully!', 'success');
    }
});

/**
 * AJAX: Manual sync
 */
add_action('wp_ajax_cf_analytics_manual_sync', function() {
    check_ajax_referer('cf_analytics_manual_sync', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $settings = get_option('cf_analytics_settings', array());
    
    if (empty($settings['zone_id'])) {
        wp_send_json_error('No zone configured');
    }
    
    $api = new CF_API();
    $zone_id = $settings['zone_id'];
    
    $end_time = gmdate('Y-m-d\TH:i:s\Z');
    $start_time = gmdate('Y-m-d\TH:i:s\Z', strtotime('-2 hours'));
    
    $result = $api->get_hourly_analytics($zone_id, $start_time, $end_time);
    
    $debug_info = array();
    $debug_info[] = "<strong>Zone ID:</strong> $zone_id";
    $debug_info[] = "<strong>Time Range:</strong> $start_time to $end_time";
    $debug_info[] = "<strong>API Success:</strong> " . ($result['success'] ? '<span style="color: #46b450;">YES</span>' : '<span style="color: #dc3232;">NO</span>');
    
    if ($result['success']) {
        $row_count = count($result['data']);
        $debug_info[] = "<strong>Data Rows Returned:</strong> $row_count";
        
        if ($row_count > 0) {
            // Insert data
            $inserted = 0;
            foreach ($result['data'] as $row) {
                $data = array(
                    'requests' => $row['sum']['requests'] ?? 0,
                    'unique_visitors' => $row['uniq']['uniques'] ?? 0,
                    'page_views' => $row['sum']['pageViews'] ?? 0,
                    'bandwidth_bytes' => $row['sum']['bytes'] ?? 0,
                    'cached_requests' => $row['sum']['cachedRequests'] ?? 0,
                    'cached_bytes' => $row['sum']['cachedBytes'] ?? 0,
                    'threats' => $row['sum']['threats'] ?? 0
                );
                
                // Convert ISO 8601 to MySQL datetime format
                $datetime_iso = $row['dimensions']['datetime'];
                $datetime_mysql = gmdate('Y-m-d H:i:s', strtotime($datetime_iso));
                
                CF_Database::insert_hourly($zone_id, $datetime_mysql, $data);
                $inserted++;
            }
            $debug_info[] = "<strong>Rows Inserted:</strong> $inserted";
            
            // Show sample data
            $sample = $result['data'][0];
            $debug_info[] = "<strong>Sample Data:</strong>";
            $debug_info[] = "• Time: " . $sample['dimensions']['datetime'];
            $debug_info[] = "• Requests: " . number_format($sample['sum']['requests']);
            $debug_info[] = "• Visitors: " . $sample['uniq']['uniques'];
            $debug_info[] = "• Bandwidth: " . CF_API::format_bytes($sample['sum']['bytes']);
            
            // Check total data in database
            global $wpdb;
            $table = $wpdb->prefix . 'cf_analytics_hourly';
            $total_rows = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE zone_id = %s",
                $zone_id
            ));
            $debug_info[] = "<strong>Total Hourly Records in DB:</strong> $total_rows";
        } else {
            $debug_info[] = '<span style="color: #f39c12;">⚠️ No data returned (zone may have no traffic in this time range)</span>';
        }
        
        update_option('cf_analytics_last_sync', current_time('mysql'));
    } else {
        $debug_info[] = "<strong>Error:</strong> " . ($result['message'] ?? 'Unknown error');
        if (isset($result['errors'])) {
            $debug_info[] = "<strong>GraphQL Errors:</strong> <pre>" . print_r($result['errors'], true) . "</pre>";
        }
    }
    
    wp_send_json_success(array(
        'message' => 'Sync completed',
        'debug' => implode('<br>', $debug_info)
    ));
});

/**
 * AJAX: Manual daily sync
 */
add_action('wp_ajax_cf_analytics_manual_daily_sync', function() {
    check_ajax_referer('cf_analytics_manual_daily_sync', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $settings = get_option('cf_analytics_settings', array());
    
    if (empty($settings['zone_id'])) {
        wp_send_json_error('No zone configured');
    }
    
    $zone_id = $settings['zone_id'];
    $api = new CF_API();
    
    // Get yesterday's date
    $yesterday = date('Y-m-d', strtotime('-1 day'));
    $start_datetime = $yesterday . 'T00:00:00Z';
    $end_datetime = $yesterday . 'T23:59:59Z';
    
    $debug_info = array();
    $debug_info[] = "<strong>Syncing Daily Data for:</strong> $yesterday";
    
    // Sync Top URLs
    $url_result = $api->get_top_urls($zone_id, $start_datetime, $end_datetime);
    if ($url_result['success'] && !empty($url_result['data'])) {
        $url_count = 0;
        foreach ($url_result['data'] as $row) {
            $url_path = $row['dimensions']['clientRequestPath'] ?? '/';
            CF_Database::insert_top_url($zone_id, $yesterday, $url_path, array(
                'requests' => $row['sum']['requests'] ?? 0,
                'visits' => $row['uniq']['uniques'] ?? 0,
                'bandwidth_bytes' => $row['sum']['bytes'] ?? 0
            ));
            $url_count++;
        }
        $debug_info[] = "<strong>✓ URLs:</strong> Synced $url_count URLs";
    } else {
        $debug_info[] = "<strong>✗ URLs:</strong> Failed or no data";
    }
    
    // Sync Countries
    $country_result = $api->get_country_data($zone_id, $start_datetime, $end_datetime);
    
    if ($country_result['success'] && !empty($country_result['data'])) {
        $country_count = 0;
        foreach ($country_result['data'] as $row) {
            $country_code = $row['dimensions']['clientCountryName'] ?? 'XX';
            $country_name = CF_API::get_country_name($country_code);
            CF_Database::insert_country($zone_id, $yesterday, $country_code, $country_name, array(
                'requests' => $row['count'] ?? 0,
                'bandwidth_bytes' => 0,
                'threats' => 0
            ));
            $country_count++;
        }
        $debug_info[] = "<strong>✓ Countries:</strong> Synced $country_count countries";
    } else {
        $debug_info[] = "<strong>✗ Countries:</strong> Failed or no data";
    }
    
    // Sync Status Codes
    $status_result = $api->get_http_status_codes($zone_id, $start_datetime, $end_datetime, 50);
    if ($status_result['success'] && !empty($status_result['data'])) {
        $status_count = 0;
        foreach ($status_result['data'] as $row) {
            $status_code = $row['dimensions']['edgeResponseStatus'] ?? 0;
            CF_Database::insert_status_code($zone_id, $yesterday, $status_code, $row['count'] ?? 0);
            $status_count++;
        }
        $debug_info[] = "<strong>✓ Status Codes:</strong> Synced $status_count codes";
    } else {
        $debug_info[] = "<strong>✗ Status Codes:</strong> Failed or no data";
    }
    
    // Sync Device Types
    $device_result = $api->get_device_types($zone_id, $start_datetime, $end_datetime);
    if ($device_result['success'] && !empty($device_result['data'])) {
        $device_count = 0;
        foreach ($device_result['data'] as $row) {
            $device_type = $row['dimensions']['clientDeviceType'] ?? 'unknown';
            CF_Database::insert_device_type($zone_id, $yesterday, $device_type, $row['count'] ?? 0);
            $device_count++;
        }
        $debug_info[] = "<strong>✓ Device Types:</strong> Synced $device_count device types";
    } else {
        $debug_info[] = "<strong>✗ Device Types:</strong> Failed or no data";
    }
    
    // Sync User Agents
    $agent_result = $api->get_user_agents($zone_id, $start_datetime, $end_datetime, 50);
    if ($agent_result['success'] && !empty($agent_result['data'])) {
        $agent_count = 0;
        foreach ($agent_result['data'] as $row) {
            $user_agent = $row['dimensions']['userAgent'] ?? 'Unknown';
            CF_Database::insert_user_agent($zone_id, $yesterday, $user_agent, $row['count'] ?? 0);
            $agent_count++;
        }
        $debug_info[] = "<strong>✓ User Agents:</strong> Synced $agent_count user agents";
    } else {
        $debug_info[] = "<strong>✗ User Agents:</strong> Failed or no data";
    }
    
    wp_send_json_success(array(
        'message' => 'Daily sync completed',
        'debug' => implode('<br>', $debug_info)
    ));
});

/**
 * AJAX: Backfill historical data
 */
add_action('wp_ajax_cf_analytics_backfill', function() {
    check_ajax_referer('cf_analytics_backfill', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $days = isset($_POST['days']) ? intval($_POST['days']) : 7;
    $days = min(max($days, 1), 30); // Between 1-30 days
    
    $result = CF_Cron::backfill_data($days);
    
    if ($result['success']) {
        wp_send_json_success($result);
    } else {
        wp_send_json_error($result['message']);
    }
});

/**
 * AJAX: Get zones list with auto-selection
 */
add_action('wp_ajax_cf_analytics_get_zones', function() {
    check_ajax_referer('cf_analytics_get_zones', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
    }
    
    $api = new CF_API();
    $result = $api->get_zones();
    
    if ($result['success']) {
        // Get WordPress site domain for auto-matching
        $site_url = parse_url(get_site_url(), PHP_URL_HOST);
        $site_domain = str_replace('www.', '', $site_url); // Remove www. for matching
        
        // Find matching zone
        $matched_zone = null;
        foreach ($result['data'] as $zone) {
            $zone_domain = str_replace('www.', '', $zone['name']);
            if ($zone_domain === $site_domain) {
                $matched_zone = $zone['id'];
                break;
            }
        }
        
        wp_send_json_success(array(
            'zones' => $result['data'],
            'matched_zone' => $matched_zone,
            'site_domain' => $site_domain
        ));
    } else {
        wp_send_json_error($result['message']);
    }
});

/**
 * Settings page HTML
 */
function cf_analytics_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    $settings = get_option('cf_analytics_settings', array());
    $is_configured = !empty($settings['email']) && !empty($settings['api_key']);
    
    if ($is_configured) {
        $email_masked = substr(CF_Encryption::decrypt($settings['email']), 0, 3) . '***@***.com';
        $api_key_masked = '******************';
    }
    
    ?>
    <div class="wrap">
        <h1><span class="dashicons dashicons-cloud" style="color: #f38020;"></span> Cloudflare Analytics - Settings</h1>
        
        <?php settings_errors('cf_analytics'); ?>
        
        <?php if (!$is_configured): ?>
            
            <div class="card" style="max-width: 800px;">
                <h2>API Configuration</h2>
                <p>Connect your Cloudflare account to start tracking analytics.</p>
                
                <form method="post">
                    <?php wp_nonce_field('cf_analytics_settings'); ?>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="cf_email">Cloudflare Email</label>
                            </th>
                            <td>
                                <input type="email" 
                                       id="cf_email" 
                                       name="cf_email" 
                                       class="regular-text" 
                                       required 
                                       placeholder="your-email@example.com">
                                <p class="description">Your Cloudflare account email address</p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="cf_api_key">Global API Key</label>
                            </th>
                            <td>
                                <input type="password" 
                                       id="cf_api_key" 
                                       name="cf_api_key" 
                                       class="regular-text" 
                                       required>
                                <p class="description">
                                    Found in your <a href="https://dash.cloudflare.com/profile/api-tokens" target="_blank">Cloudflare Profile → API Tokens</a>
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <label for="cf_zone_id">Zone ID</label>
                            </th>
                            <td>
                                <input type="text" 
                                       id="cf_zone_id" 
                                       name="cf_zone_id" 
                                       class="regular-text" 
                                       required 
                                       value="<?php echo esc_attr($settings['zone_id'] ?? ''); ?>">
                                <button type="button" class="button" id="cf-load-zones" style="margin-left: 10px;">
                                    Load My Zones
                                </button>
                                <p class="description">
                                    Your Cloudflare zone ID (found in dashboard overview)
                                </p>
                                <div id="cf-zones-list" style="display: none; margin-top: 10px; padding: 10px; background: #f0f0f1; border-radius: 4px;"></div>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <button type="submit" name="cf_analytics_save_settings" class="button button-primary button-large">
                            Connect to Cloudflare
                        </button>
                    </p>
                </form>
            </div>
            
        <?php else: ?>
            
            <!-- Connected Status -->
            <div class="card" style="max-width: 800px;">
                <h2>Connection Status</h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Status</th>
                        <td>
                            <span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
                            <strong style="color: #46b450;">Connected</strong>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">Email</th>
                        <td><?php echo esc_html($email_masked); ?></td>
                    </tr>
                    <tr>
                        <th scope="row">API Key</th>
                        <td><?php echo esc_html($api_key_masked); ?></td>
                    </tr>
                    <tr>
                        <th scope="row">Zone ID</th>
                        <td><code><?php echo esc_html($settings['zone_id']); ?></code></td>
                    </tr>
                    <tr>
                        <th scope="row">Last Sync</th>
                        <td>
                            <?php 
                            $last_sync = get_option('cf_analytics_last_sync');
                            echo $last_sync ? esc_html($last_sync) : 'Never';
                            ?>
                        </td>
                    </tr>
                </table>
                
                <div style="margin-top: 20px; padding-top: 20px; border-top: 1px solid #ddd;">
                    <h4 style="margin: 0 0 10px 0;">Quick Sync</h4>
                    <button type="button" class="button" id="cf-manual-sync">
                        <span class="dashicons dashicons-update"></span> Sync Hourly Data
                    </button>
                    <button type="button" class="button" id="cf-manual-daily-sync" style="margin-left: 10px;">
                        <span class="dashicons dashicons-backup"></span> Sync All Daily Data
                    </button>
                    <span class="spinner" style="float: none; margin: 0 10px;"></span>
                    <span id="cf-sync-result"></span>
                    <div id="cf-sync-debug" style="margin-top: 15px; padding: 15px; background: #f0f0f1; border-radius: 4px; display: none; font-size: 13px; line-height: 1.8;"></div>
                </div>
            </div>
            
            <!-- Backfill Data -->
            <div class="card" style="max-width: 800px; margin-top: 20px;">
                <h2><span class="dashicons dashicons-backup"></span> Backfill Historical Data</h2>
                <p>Pull historical analytics data from Cloudflare (up to 30 days).</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="backfill-days">Number of Days</label>
                        </th>
                        <td>
                            <select id="backfill-days" class="regular-text">
                                <option value="7">7 days</option>
                                <option value="14">14 days</option>
                                <option value="30">30 days (maximum)</option>
                            </select>
                            <p class="description">This will fetch hourly metrics + top URLs/countries/browsers for each day.</p>
                        </td>
                    </tr>
                </table>
                
                <button type="button" class="button button-secondary" id="cf-backfill-btn">
                    <span class="dashicons dashicons-download"></span> Start Backfill
                </button>
                <span class="spinner" style="float: none; margin: 0 10px;" id="backfill-spinner"></span>
                <div id="backfill-result" style="margin-top: 10px;"></div>
            </div>
            
            <!-- Database Statistics -->
            <div class="card" style="max-width: 800px; margin-top: 20px;">
                <h2><span class="dashicons dashicons-database"></span> Database Statistics</h2>
                
                <?php
                $stats = CF_Database::get_database_size();
                ?>
                
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>Table</th>
                            <th>Rows</th>
                            <th>Size</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stats as $table => $data): ?>
                            <?php if ($table === 'total') continue; ?>
                            <tr>
                                <td><code><?php echo esc_html($table); ?></code></td>
                                <td><?php echo number_format($data['rows']); ?></td>
                                <td><?php echo CF_API::format_bytes($data['size']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <tr style="font-weight: bold; background: #f0f0f1;">
                            <td>Total</td>
                            <td><?php echo number_format($stats['total']['rows']); ?> rows</td>
                            <td><?php echo CF_API::format_bytes($stats['total']['size']); ?></td>
                        </tr>
                    </tbody>
                </table>
                
                <p class="description" style="margin-top: 15px;">
                    <strong>Storage Efficiency:</strong> Hourly data kept for 7 days (~7 KB). Daily data kept permanently (~16 KB/year). 
                    URL/Country/Browser data kept for 30 days. After 10 years: ~335 KB total.
                </p>
            </div>
            
            <!-- Danger Zone -->
            <div class="card" style="max-width: 800px; margin-top: 20px; border-left: 4px solid #dc3232;">
                <h2 style="color: #dc3232;"><span class="dashicons dashicons-warning"></span> Disconnect</h2>
                <p>Remove Cloudflare API credentials. This will not delete collected data.</p>
                
                <form method="post" onsubmit="return confirm('Are you sure you want to disconnect?');">
                    <?php wp_nonce_field('cf_analytics_disconnect'); ?>
                    <button type="submit" name="cf_analytics_disconnect" class="button button-link-delete">
                        Disconnect from Cloudflare
                    </button>
                </form>
            </div>
            
        <?php endif; ?>
    </div>
    
    <script>
    jQuery(document).ready(function($) {
        // Load zones
        $('#cf-load-zones').on('click', function() {
            var email = $('#cf_email').val();
            var apiKey = $('#cf_api_key').val();
            
            if (!email || !apiKey) {
                alert('Please enter your email and API key first.');
                return;
            }
            
            var $btn = $(this);
            $btn.prop('disabled', true).text('Loading...');
            
            $.post(ajaxurl, {
                action: 'cf_analytics_get_zones',
                nonce: '<?php echo wp_create_nonce('cf_analytics_get_zones'); ?>'
            }, function(response) {
                $btn.prop('disabled', false).text('Load My Zones');
                
                if (response.success) {
                    var zones = response.data.zones;
                    var matchedZone = response.data.matched_zone;
                    var siteDomain = response.data.site_domain;
                    
                    var html = '<p><strong>Select your zone:</strong>';
                    if (matchedZone) {
                        html += ' <span style="color: #46b450; margin-left: 10px;">✓ Auto-detected: ' + siteDomain + '</span>';
                    }
                    html += '</p><ul style="list-style: none; padding: 0;">';
                    
                    zones.forEach(function(zone) {
                        var isMatched = (zone.id === matchedZone);
                        var style = isMatched ? 'background: #d4edda; padding: 8px; border-radius: 4px; border: 2px solid #46b450;' : '';
                        var checked = isMatched ? 'checked' : '';
                        
                        html += '<li style="margin: 5px 0; ' + style + '">';
                        html += '<label style="cursor: pointer; display: block;">';
                        html += '<input type="radio" name="zone_select" value="' + zone.id + '" ' + checked + '> ';
                        html += '<strong>' + zone.name + '</strong> ';
                        html += '<code style="margin-left: 10px;">' + zone.id + '</code>';
                        if (isMatched) {
                            html += ' <span style="color: #46b450; margin-left: 10px;">← Current Site</span>';
                        }
                        html += '</label></li>';
                    });
                    html += '</ul>';
                    
                    $('#cf-zones-list').html(html).show();
                    
                    // Auto-fill zone ID if matched
                    if (matchedZone) {
                        $('#cf_zone_id').val(matchedZone);
                    }
                    
                    $(document).on('change', 'input[name="zone_select"]', function() {
                        $('#cf_zone_id').val($(this).val());
                    });
                } else {
                    alert('Failed to load zones: ' + response.data);
                }
            });
        });
        
        // Manual sync
        $('#cf-manual-sync').on('click', function() {
            var $btn = $(this);
            var $spinner = $btn.next('.spinner');
            var $result = $('#cf-sync-result');
            var $debug = $('#cf-sync-debug');
            
            $btn.prop('disabled', true);
            $spinner.addClass('is-active');
            $result.html('');
            $debug.hide().html('');
            
            $.post(ajaxurl, {
                action: 'cf_analytics_manual_sync',
                nonce: '<?php echo wp_create_nonce('cf_analytics_manual_sync'); ?>'
            }, function(response) {
                $btn.prop('disabled', false);
                $spinner.removeClass('is-active');
                
                if (response.success) {
                    $result.html('<span style="color: #46b450;">✓ Sync completed!</span>');
                    if (response.data.debug) {
                        $debug.html(response.data.debug).show();
                    }
                } else {
                    $result.html('<span style="color: #dc3232;">✗ ' + response.data + '</span>');
                }
            });
        });
        
        // Manual daily sync
        $('#cf-manual-daily-sync').on('click', function() {
            var $btn = $(this);
            var $spinner = $('.spinner');
            var $result = $('#cf-sync-result');
            var $debug = $('#cf-sync-debug');
            
            $btn.prop('disabled', true);
            $('#cf-manual-sync').prop('disabled', true);
            $spinner.addClass('is-active');
            $result.html('');
            $debug.hide().html('');
            
            $.post(ajaxurl, {
                action: 'cf_analytics_manual_daily_sync',
                nonce: '<?php echo wp_create_nonce('cf_analytics_manual_daily_sync'); ?>'
            }, function(response) {
                $btn.prop('disabled', false);
                $('#cf-manual-sync').prop('disabled', false);
                $spinner.removeClass('is-active');
                
                if (response.success) {
                    $result.html('<span style="color: #46b450;">✓ Daily sync completed!</span>');
                    if (response.data.debug) {
                        $debug.html(response.data.debug).show();
                    }
                } else {
                    $result.html('<span style="color: #dc3232;">✗ ' + response.data + '</span>');
                }
            });
        });
        
        // Backfill
        $('#cf-backfill-btn').on('click', function() {
            var days = $('#backfill-days').val();
            var $btn = $(this);
            var $spinner = $('#backfill-spinner');
            var $result = $('#backfill-result');
            
            if (!confirm('This will fetch ' + days + ' days of historical data. This may take a few minutes. Continue?')) {
                return;
            }
            
            $btn.prop('disabled', true);
            $spinner.addClass('is-active');
            $result.html('<p style="color: #666;">⏳ Fetching data... This may take a few minutes. Please do not close this page.</p>');
            
            $.post(ajaxurl, {
                action: 'cf_analytics_backfill',
                nonce: '<?php echo wp_create_nonce('cf_analytics_backfill'); ?>',
                days: days
            }, function(response) {
                $btn.prop('disabled', false);
                $spinner.removeClass('is-active');
                
                if (response.success) {
                    $result.html('<div class="notice notice-success inline"><p><strong>Success!</strong> ' + response.data.message + '</p></div>');
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    $result.html('<div class="notice notice-error inline"><p><strong>Error:</strong> ' + response.data + '</p></div>');
                }
            });
        });
    });
    </script>
    
    <style>
    .card {
        background: #fff;
        border: 1px solid #c3c4c7;
        border-radius: 4px;
        padding: 20px;
        box-shadow: 0 1px 1px rgba(0,0,0,.04);
    }
    .card h2 {
        margin-top: 0;
        font-size: 18px;
    }
    </style>
    <?php
}
