<?php
/**
 * Analytics dashboard page with tabs
 */

if (!defined('ABSPATH')) {
    exit;
}

function cf_analytics_page() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    $settings = get_option('cf_analytics_settings', array());
    
    if (empty($settings['zone_id'])) {
        ?>
        <div class="wrap">
            <h1>Cloudflare Analytics</h1>
            <div class="notice notice-warning">
                <p><strong>Setup Required:</strong> Please <a href="<?php echo admin_url('admin.php?page=cf-analytics-settings'); ?>">configure your Cloudflare API credentials</a> to view analytics.</p>
            </div>
        </div>
        <?php
        return;
    }
    
    $zone_id = $settings['zone_id'];
    $current_tab = isset($_GET['tab']) ? $_GET['tab'] : 'overview';
    
    // Enqueue Chart.js in header so it's available
    wp_enqueue_script('chartjs', 'https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js', array(), '4.4.0', false);
    wp_enqueue_style('cf-analytics-admin', CF_ANALYTICS_PLUGIN_URL . 'assets/css/admin.css', array(), CF_ANALYTICS_VERSION);
    
    ?>
    <div class="wrap">
        <h1 class="wp-heading-inline">
            <span class="dashicons dashicons-cloud" style="color: #f38020;"></span> Cloudflare Analytics Dashboard
        </h1>
        
        <nav class="nav-tab-wrapper wp-clearfix" style="margin: 20px 0;">
            <a href="?page=cf-analytics&tab=overview" class="nav-tab <?php echo $current_tab === 'overview' ? 'nav-tab-active' : ''; ?>">
                <span class="dashicons dashicons-chart-line"></span> Overview
            </a>
            <a href="?page=cf-analytics&tab=urls" class="nav-tab <?php echo $current_tab === 'urls' ? 'nav-tab-active' : ''; ?>">
                <span class="dashicons dashicons-admin-links"></span> Top Content
            </a>
            <a href="?page=cf-analytics&tab=geography" class="nav-tab <?php echo $current_tab === 'geography' ? 'nav-tab-active' : ''; ?>">
                <span class="dashicons dashicons-admin-site"></span> Geography
            </a>
            <a href="?page=cf-analytics&tab=traffic" class="nav-tab <?php echo $current_tab === 'traffic' ? 'nav-tab-active' : ''; ?>">
                <span class="dashicons dashicons-networking"></span> Traffic Analysis
            </a>
            <a href="?page=cf-analytics&tab=bots" class="nav-tab <?php echo $current_tab === 'bots' ? 'nav-tab-active' : ''; ?>">
                <span class="dashicons dashicons-search"></span> Bots & Crawlers
            </a>
        </nav>
        
        <?php
        switch ($current_tab) {
            case 'overview':
                render_overview_tab($zone_id);
                break;
            case 'urls':
                render_urls_tab($zone_id);
                break;
            case 'geography':
                render_geography_tab($zone_id);
                break;
            case 'traffic':
                render_traffic_tab($zone_id);
                break;
            case 'bots':
                render_bots_tab($zone_id);
                break;
        }
        ?>
    </div>
    <?php
}

/**
 * Overview Tab - Charts and metrics
 */
function render_overview_tab($zone_id) {
    // Get period from query string (default: today)
    $period = isset($_GET['period']) ? $_GET['period'] : 'today';
    
    $periods = array(
        'today' => array('label' => 'Today', 'start' => gmdate('Y-m-d') . ' 00:00:00', 'end' => gmdate('Y-m-d H:i:s')),
        'yesterday' => array('label' => 'Yesterday', 'start' => gmdate('Y-m-d', strtotime('-1 day')) . ' 00:00:00', 'end' => gmdate('Y-m-d', strtotime('-1 day')) . ' 23:59:59'),
        '7days' => array('label' => 'Last 7 Days', 'start' => gmdate('Y-m-d H:i:s', strtotime('-7 days')), 'end' => gmdate('Y-m-d H:i:s')),
        '30days' => array('label' => 'Last 30 Days', 'start' => gmdate('Y-m-d H:i:s', strtotime('-30 days')), 'end' => gmdate('Y-m-d H:i:s')),
        'year' => array('label' => 'Last Year', 'start' => gmdate('Y-m-d H:i:s', strtotime('-365 days')), 'end' => gmdate('Y-m-d H:i:s')),
        'alltime' => array('label' => 'All Time', 'start' => '2020-01-01 00:00:00', 'end' => gmdate('Y-m-d H:i:s'))
    );
    
    $current_period = $periods[$period] ?? $periods['7days'];
    $start_date = $current_period['start'];
    $end_date = $current_period['end'];
    
    // Get earliest data date
    global $wpdb;
    $earliest_date = $wpdb->get_var($wpdb->prepare(
        "SELECT MIN(date) FROM {$wpdb->prefix}cf_analytics_daily WHERE zone_id = %s",
        $zone_id
    ));
    
    // Get data - use daily for year/alltime, hourly for shorter periods
    if ($period === 'year' || $period === 'alltime') {
        // Use daily aggregates for long periods
        $daily_data = CF_Database::get_daily_data($zone_id, $start_date, $end_date);
        $totals = CF_Database::get_period_totals($zone_id, $start_date, $end_date);
        
        $labels = array();
        $requests_data = array();
        $bandwidth_data = array();
        $cached_data = array();
        $uncached_data = array();
        
        foreach ($daily_data as $row) {
            $labels[] = date('M j', strtotime($row->date));
            $requests_data[] = $row->requests;
            $bandwidth_data[] = round($row->bandwidth_bytes / 1048576, 2); // MB
            $cached_data[] = $row->cached_requests;
            $uncached_data[] = $row->requests - $row->cached_requests;
        }
    } else {
        // Use hourly data for short periods
        $hourly_data = CF_Database::get_hourly_data($zone_id, $start_date, $end_date);
        $totals = CF_Database::get_period_totals($zone_id, $start_date, $end_date);
        
        $labels = array();
        $requests_data = array();
        $bandwidth_data = array();
        $cached_data = array();
        $uncached_data = array();
        
        foreach ($hourly_data as $row) {
            // Convert UTC to WordPress timezone for display
            $utc_datetime = strtotime($row->datetime . ' UTC');
            $local_datetime = get_date_from_gmt(gmdate('Y-m-d H:i:s', $utc_datetime), 'Y-m-d H:i:s');
            
            $labels[] = date('M j, g A', strtotime($local_datetime));
            $requests_data[] = $row->requests;
            $bandwidth_data[] = round($row->bandwidth_bytes / 1048576, 2); // MB
            $cached_data[] = $row->cached_requests;
            $uncached_data[] = $row->requests - $row->cached_requests;
        }
    }
    
    ?>
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
        <div class="cf-period-selector">
            <a href="?page=cf-analytics&tab=overview&period=today" class="button <?php echo $period === 'today' ? 'button-primary' : ''; ?>">Today</a>
            <a href="?page=cf-analytics&tab=overview&period=yesterday" class="button <?php echo $period === 'yesterday' ? 'button-primary' : ''; ?>">Yesterday</a>
            <a href="?page=cf-analytics&tab=overview&period=7days" class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Last 7 Days</a>
            <a href="?page=cf-analytics&tab=overview&period=30days" class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Last 30 Days</a>
            <a href="?page=cf-analytics&tab=overview&period=year" class="button <?php echo $period === 'year' ? 'button-primary' : ''; ?>">Last Year</a>
            <a href="?page=cf-analytics&tab=overview&period=alltime" class="button <?php echo $period === 'alltime' ? 'button-primary' : ''; ?>">All Time</a>
        </div>
        
        <?php if ($earliest_date): ?>
        <div style="color: #666; font-size: 13px;">
            <span class="dashicons dashicons-calendar-alt" style="font-size: 14px; margin-top: 2px;"></span>
            Data tracking since: <strong><?php echo date('M j, Y', strtotime($earliest_date)); ?></strong>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Stats Cards -->
    <div class="cf-stats-grid">
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                <span class="dashicons dashicons-chart-line"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Total Requests</div>
                <div class="cf-stat-value"><?php echo CF_API::format_number($totals->total_requests ?? 0); ?></div>
            </div>
        </div>
        
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                <span class="dashicons dashicons-groups"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Unique Visitors</div>
                <div class="cf-stat-value"><?php echo CF_API::format_number($totals->total_visitors ?? 0); ?></div>
            </div>
        </div>
        
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);">
                <span class="dashicons dashicons-download"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Bandwidth</div>
                <div class="cf-stat-value"><?php echo CF_API::format_bytes($totals->total_bandwidth ?? 0); ?></div>
            </div>
        </div>
        
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%);">
                <span class="dashicons dashicons-performance"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Cache Hit Rate</div>
                <div class="cf-stat-value"><?php echo round($totals->cache_hit_rate ?? 0, 1); ?>%</div>
            </div>
        </div>
        
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%);">
                <span class="dashicons dashicons-shield"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Threats Blocked</div>
                <div class="cf-stat-value"><?php echo CF_API::format_number($totals->total_threats ?? 0); ?></div>
            </div>
        </div>
        
        <?php
        // Get device breakdown for selected period
        $device_data = CF_Database::get_device_types($zone_id, date('Y-m-d', strtotime($start_date)), date('Y-m-d', strtotime($end_date)));
        if (!empty($device_data)):
            $total_device_requests = array_sum(array_map(function($d) { return $d->total_requests; }, $device_data));
            $desktop_pct = 0;
            $mobile_pct = 0;
            foreach ($device_data as $d) {
                if ($d->device_type === 'desktop') $desktop_pct = round(($d->total_requests / $total_device_requests) * 100);
                if ($d->device_type === 'mobile') $mobile_pct = round(($d->total_requests / $total_device_requests) * 100);
            }
        ?>
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);">
                <span class="dashicons dashicons-smartphone"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Device Split</div>
                <div class="cf-stat-value" style="font-size: 14px;"><?php echo $desktop_pct; ?>% Desktop / <?php echo $mobile_pct; ?>% Mobile</div>
            </div>
        </div>
        <?php endif; ?>
        
        <?php
        // Get top user agents for selected period
        $top_agents = CF_Database::get_user_agents($zone_id, date('Y-m-d', strtotime($start_date)), date('Y-m-d', strtotime($end_date)), 5);
        if (!empty($top_agents)):
            $bot_count = 0;
            foreach ($top_agents as $agent) {
                if (stripos($agent->user_agent, 'bot') !== false || stripos($agent->user_agent, 'crawler') !== false) {
                    $bot_count++;
                }
            }
        ?>
        <div class="cf-stat-card">
            <div class="cf-stat-icon" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);">
                <span class="dashicons dashicons-search"></span>
            </div>
            <div class="cf-stat-content">
                <div class="cf-stat-label">Top 5 Visitors</div>
                <div class="cf-stat-value" style="font-size: 18px;"><?php echo $bot_count; ?> Bots / <?php echo (5 - $bot_count); ?> Browsers</div>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Charts -->
    <div class="cf-charts-container">
        <div class="cf-chart-box">
            <h3>Requests Over Time</h3>
            <canvas id="requestsChart"></canvas>
        </div>
        
        <div class="cf-chart-box">
            <h3>Bandwidth Usage (MB)</h3>
            <canvas id="bandwidthChart"></canvas>
        </div>
        
        <?php
        // Get device types for selected period
        $overview_devices = CF_Database::get_device_types($zone_id, date('Y-m-d', strtotime($start_date)), date('Y-m-d', strtotime($end_date)));
        ?>
        <?php if (!empty($overview_devices)): ?>
        <div class="cf-chart-box">
            <h3>Device Types (<?php echo $current_period['label']; ?>)</h3>
            <canvas id="overviewDevicesChart"></canvas>
        </div>
        <?php endif; ?>
        
        <?php
        // Get top user agents for selected period
        $overview_agents = CF_Database::get_user_agents($zone_id, date('Y-m-d', strtotime($start_date)), date('Y-m-d', strtotime($end_date)), 5);
        ?>
        <?php if (!empty($overview_agents)): ?>
        <div class="cf-chart-box">
            <h3>Top 5 User Agents (<?php echo $current_period['label']; ?>)</h3>
            <canvas id="overviewBotsChart"></canvas>
        </div>
        <?php endif; ?>
        
        <?php
        // Get top URLs for selected period
        $overview_urls = CF_Database::get_top_content_urls($zone_id, date('Y-m-d', strtotime($start_date)), date('Y-m-d', strtotime($end_date)), 5);
        ?>
        <?php if (!empty($overview_urls)): ?>
        <div class="cf-chart-box">
            <h3>Top 5 Pages (<?php echo $current_period['label']; ?>)</h3>
            <canvas id="overviewUrlsChart"></canvas>
        </div>
        <?php endif; ?>
        
        <div class="cf-chart-box" style="grid-column: 1 / -1;">
            <h3>Cache Performance</h3>
            <canvas id="cacheChart"></canvas>
        </div>
    </div>
    
    <script>
    window.addEventListener('load', function() {
        // Wait for Chart.js to be fully loaded
        if (typeof Chart === 'undefined') {
            console.error('Chart.js not loaded');
            return;
        }
        
        const chartColors = {
            primary: 'rgba(102, 126, 234, 1)',
            primaryLight: 'rgba(102, 126, 234, 0.1)',
            secondary: 'rgba(79, 172, 254, 1)',
            secondaryLight: 'rgba(79, 172, 254, 0.1)',
            success: 'rgba(67, 233, 123, 1)',
            successLight: 'rgba(67, 233, 123, 0.1)',
            warning: 'rgba(254, 225, 64, 1)',
            warningLight: 'rgba(254, 225, 64, 0.1)'
        };
        
        // Requests Chart
        new Chart(document.getElementById('requestsChart'), {
        type: 'line',
        data: {
            labels: <?php echo json_encode($labels); ?>,
            datasets: [{
                label: 'Requests',
                data: <?php echo json_encode($requests_data); ?>,
                borderColor: chartColors.primary,
                backgroundColor: chartColors.primaryLight,
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { display: false }
            },
            scales: {
                y: { beginAtZero: true }
            }
        }
    });
    
    // Bandwidth Chart
    new Chart(document.getElementById('bandwidthChart'), {
        type: 'line',
        data: {
            labels: <?php echo json_encode($labels); ?>,
            datasets: [{
                label: 'Bandwidth (MB)',
                data: <?php echo json_encode($bandwidth_data); ?>,
                borderColor: chartColors.secondary,
                backgroundColor: chartColors.secondaryLight,
                fill: true,
                tension: 0.4
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { display: false }
            },
            scales: {
                y: { beginAtZero: true }
            }
        }
    });
    
    // Cache Chart
    new Chart(document.getElementById('cacheChart'), {
        type: 'bar',
        data: {
            labels: <?php echo json_encode($labels); ?>,
            datasets: [
                {
                    label: 'Cached',
                    data: <?php echo json_encode($cached_data); ?>,
                    backgroundColor: chartColors.success
                },
                {
                    label: 'Uncached',
                    data: <?php echo json_encode($uncached_data); ?>,
                    backgroundColor: chartColors.warning
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: { stacked: true },
                y: { stacked: true, beginAtZero: true }
            }
        }
    });
    
    <?php if (!empty($overview_devices)): ?>
    // Device Types Chart for Overview
    <?php
    $ov_device_labels = array();
    $ov_device_values = array();
    foreach ($overview_devices as $item) {
        $ov_device_labels[] = ucfirst($item->device_type);
        $ov_device_values[] = (int)$item->total_requests;
    }
    ?>
    new Chart(document.getElementById('overviewDevicesChart'), {
        type: 'doughnut',
        data: {
            labels: <?php echo json_encode($ov_device_labels); ?>,
            datasets: [{
                data: <?php echo json_encode($ov_device_values); ?>,
                backgroundColor: ['#667eea', '#f093fb', '#4facfe']
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { position: 'bottom' }
            }
        }
    });
    <?php endif; ?>
    
    <?php if (!empty($overview_agents)): ?>
    // Top User Agents Chart for Overview
    <?php
    $ov_agent_labels = array();
    $ov_agent_values = array();
    $ov_agent_colors = array();
    foreach ($overview_agents as $item) {
        $agent = $item->user_agent;
        // Shorten user agent for display
        $short_name = strlen($agent) > 30 ? substr($agent, 0, 27) . '...' : $agent;
        $ov_agent_labels[] = $short_name;
        $ov_agent_values[] = (int)$item->total_requests;
        
        // Color based on type
        if (stripos($agent, 'bot') !== false || stripos($agent, 'crawler') !== false) {
            $ov_agent_colors[] = '#f093fb'; // Bot - pink
        } else {
            $ov_agent_colors[] = '#667eea'; // Browser - purple
        }
    }
    ?>
    new Chart(document.getElementById('overviewBotsChart'), {
        type: 'bar',
        data: {
            labels: <?php echo json_encode($ov_agent_labels); ?>,
            datasets: [{
                label: 'Requests',
                data: <?php echo json_encode($ov_agent_values); ?>,
                backgroundColor: <?php echo json_encode($ov_agent_colors); ?>
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            indexAxis: 'y',
            plugins: {
                legend: { display: false }
            },
            scales: {
                x: { beginAtZero: true }
            }
        }
    });
    <?php endif; ?>
    
    <?php if (!empty($overview_urls)): ?>
    // Top URLs Chart
    <?php
    $ov_url_labels = array();
    $ov_url_values = array();
    foreach ($overview_urls as $url) {
        $label = $url->url_path;
        // Truncate long URLs from the middle to keep start and end
        if (strlen($label) > 30) {
            $start = substr($label, 0, 15);
            $end = substr($label, -12);
            $label = $start . '...' . $end;
        }
        $ov_url_labels[] = $label;
        $ov_url_values[] = $url->total_requests;
    }
    ?>
    new Chart(document.getElementById('overviewUrlsChart'), {
        type: 'bar',
        data: {
            labels: <?php echo json_encode($ov_url_labels); ?>,
            datasets: [{
                label: 'Requests',
                data: <?php echo json_encode($ov_url_values); ?>,
                backgroundColor: 'rgba(102, 126, 234, 0.8)'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            indexAxis: 'y',
            plugins: {
                legend: { display: false },
                tooltip: {
                    callbacks: {
                        title: function(context) {
                            return <?php echo json_encode(array_column($overview_urls, 'url_path')); ?>[context[0].dataIndex];
                        }
                    }
                }
            },
            scales: {
                x: { beginAtZero: true },
                y: {
                    ticks: {
                        font: {
                            size: 11,
                            family: 'monospace'
                        }
                    }
                }
            }
        }
    });
    <?php endif; ?>
    
    }); // End window.addEventListener
    </script>
    <?php
}

/**
 * Top URLs Tab
 */
function render_urls_tab($zone_id) {
    $period = isset($_GET['period']) ? $_GET['period'] : 'today';
    $view = isset($_GET['view']) ? $_GET['view'] : 'content';
    
    $periods = array(
        'today' => array('label' => 'Today', 'start' => gmdate('Y-m-d'), 'end' => gmdate('Y-m-d')),
        'yesterday' => array('label' => 'Yesterday', 'start' => gmdate('Y-m-d', strtotime('-1 day')), 'end' => gmdate('Y-m-d', strtotime('-1 day'))),
        '7days' => array('label' => 'Last 7 Days', 'start' => gmdate('Y-m-d', strtotime('-7 days')), 'end' => gmdate('Y-m-d')),
        '30days' => array('label' => 'Last 30 Days', 'start' => gmdate('Y-m-d', strtotime('-30 days')), 'end' => gmdate('Y-m-d')),
        'year' => array('label' => 'Last Year', 'start' => gmdate('Y-m-d', strtotime('-365 days')), 'end' => gmdate('Y-m-d')),
        'alltime' => array('label' => 'All Time', 'start' => '2020-01-01', 'end' => gmdate('Y-m-d'))
    );
    
    $current_period = $periods[$period] ?? $periods['7days'];
    
    // Get all URLs
    $all_urls = CF_Database::get_top_urls($zone_id, $current_period['start'], $current_period['end'], 999);
    
    // Filter for WordPress content if in content view
    if ($view === 'content') {
        // Get all post/page slugs
        $content_slugs = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        ));
        
        $valid_paths = array('/'); // Homepage is always valid
        
        foreach ($content_slugs as $post_id) {
            $permalink = get_permalink($post_id);
            $path = parse_url($permalink, PHP_URL_PATH);
            if ($path) {
                $valid_paths[] = rtrim($path, '/');
            }
        }
        
        // Filter URLs to only include WordPress content
        $urls = array_filter($all_urls, function($url) use ($valid_paths) {
            $clean_path = rtrim($url->url_path, '/');
            // Exclude any URLs with double slashes (malformed/bot requests)
            if (strpos($url->url_path, '//') !== false) {
                return false;
            }
            return in_array($clean_path, $valid_paths);
        });
    } else {
        $urls = $all_urls;
    }
    
    ?>
    <!-- View Switcher -->
    <div style="margin: 20px 0; border-bottom: 2px solid #ddd;">
        <a href="?page=cf-analytics&tab=urls&period=<?php echo $period; ?>&view=content" 
           class="button <?php echo $view === 'content' ? 'button-primary' : ''; ?>" 
           style="border-radius: 4px 4px 0 0; margin-bottom: -2px;">
            📄 Content Traffic
        </a>
        <a href="?page=cf-analytics&tab=urls&period=<?php echo $period; ?>&view=all" 
           class="button <?php echo $view === 'all' ? 'button-primary' : ''; ?>" 
           style="border-radius: 4px 4px 0 0; margin-bottom: -2px;">
            🌐 All URLs
        </a>
    </div>
    
    <div class="cf-period-selector">
        <a href="?page=cf-analytics&tab=urls&period=today&view=<?php echo $view; ?>" class="button <?php echo $period === 'today' ? 'button-primary' : ''; ?>">Today</a>
        <a href="?page=cf-analytics&tab=urls&period=yesterday&view=<?php echo $view; ?>" class="button <?php echo $period === 'yesterday' ? 'button-primary' : ''; ?>">Yesterday</a>
        <a href="?page=cf-analytics&tab=urls&period=7days&view=<?php echo $view; ?>" class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Last 7 Days</a>
        <a href="?page=cf-analytics&tab=urls&period=30days&view=<?php echo $view; ?>" class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Last 30 Days</a>
        <a href="?page=cf-analytics&tab=urls&period=year&view=<?php echo $view; ?>" class="button <?php echo $period === 'year' ? 'button-primary' : ''; ?>">Last Year</a>
        <a href="?page=cf-analytics&tab=urls&period=alltime&view=<?php echo $view; ?>" class="button <?php echo $period === 'alltime' ? 'button-primary' : ''; ?>">All Time</a>
    </div>
    
    <div class="cf-table-container">
        <h2><?php echo $view === 'content' ? 'Top Content' : 'Top URLs'; ?> by Requests</h2>
        
        <?php if (empty($urls)): ?>
            <div class="notice notice-info inline">
                <p>No URL data available yet. The daily sync will populate this data starting tomorrow.</p>
            </div>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 60px;">#</th>
                        <th><?php echo $view === 'content' ? 'Content Title' : 'URL Path'; ?></th>
                        <th style="width: 120px;">Requests</th>
                        <th style="width: 120px;">Visits</th>
                        <th style="width: 120px;">Bandwidth</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $rank = 1; ?>
                    <?php foreach ($urls as $url): ?>
                        <?php
                        $display_text = $url->url_path;
                        if ($view === 'content') {
                            // Try to get post title from URL
                            $post_id = url_to_postid($url->url_path);
                            if ($post_id) {
                                $display_text = get_the_title($post_id);
                            } elseif ($url->url_path === '/' || $url->url_path === '') {
                                $display_text = 'Homepage';
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo $rank++; ?></td>
                            <td>
                                <?php if ($view === 'content'): ?>
                                    <strong><?php echo esc_html($display_text); ?></strong>
                                    <br><small style="color: #666;"><code><?php echo esc_html($url->url_path); ?></code></small>
                                <?php else: ?>
                                    <code><?php echo esc_html($url->url_path); ?></code>
                                <?php endif; ?>
                            </td>
                            <td><?php echo number_format($url->total_requests); ?></td>
                            <td><?php echo number_format($url->total_visits); ?></td>
                            <td><?php echo CF_API::format_bytes($url->total_bandwidth); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php
}

/**
 * Geography Tab
 */
function render_geography_tab($zone_id) {
    $period = isset($_GET['period']) ? $_GET['period'] : 'today';
    
    $periods = array(
        'today' => array('label' => 'Today', 'start' => gmdate('Y-m-d'), 'end' => gmdate('Y-m-d')),
        'yesterday' => array('label' => 'Yesterday', 'start' => gmdate('Y-m-d', strtotime('-1 day')), 'end' => gmdate('Y-m-d', strtotime('-1 day'))),
        '7days' => array('label' => 'Last 7 Days', 'start' => gmdate('Y-m-d', strtotime('-7 days')), 'end' => gmdate('Y-m-d')),
        '30days' => array('label' => 'Last 30 Days', 'start' => gmdate('Y-m-d', strtotime('-30 days')), 'end' => gmdate('Y-m-d')),
        'year' => array('label' => 'Last Year', 'start' => gmdate('Y-m-d', strtotime('-365 days')), 'end' => gmdate('Y-m-d')),
        'alltime' => array('label' => 'All Time', 'start' => '2020-01-01', 'end' => gmdate('Y-m-d'))
    );
    
    $current_period = $periods[$period] ?? $periods['7days'];
    $countries = CF_Database::get_top_countries($zone_id, $current_period['start'], $current_period['end'], 50);
    
    ?>
    <div class="cf-period-selector">
        <a href="?page=cf-analytics&tab=geography&period=today" class="button <?php echo $period === 'today' ? 'button-primary' : ''; ?>">Today</a>
        <a href="?page=cf-analytics&tab=geography&period=yesterday" class="button <?php echo $period === 'yesterday' ? 'button-primary' : ''; ?>">Yesterday</a>
        <a href="?page=cf-analytics&tab=geography&period=7days" class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Last 7 Days</a>
        <a href="?page=cf-analytics&tab=geography&period=30days" class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Last 30 Days</a>
        <a href="?page=cf-analytics&tab=geography&period=year" class="button <?php echo $period === 'year' ? 'button-primary' : ''; ?>">Last Year</a>
        <a href="?page=cf-analytics&tab=geography&period=alltime" class="button <?php echo $period === 'alltime' ? 'button-primary' : ''; ?>">All Time</a>
    </div>
    
    <div class="cf-table-container">
        <h2>Top Countries by Requests</h2>
        <p class="description" style="margin-top: 0;">Request counts only. Bandwidth and threat data require Cloudflare Pro plan or higher.</p>
        
        <?php if (empty($countries)): ?>
            <div class="notice notice-info inline">
                <p>No geographic data available yet. Click "Sync Daily Data" in Settings to populate this data.</p>
            </div>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 60px;">#</th>
                        <th style="width: 80px;">Code</th>
                        <th>Country</th>
                        <th style="width: 150px;">Requests</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $rank = 1; ?>
                    <?php foreach ($countries as $country): ?>
                        <tr>
                            <td><?php echo $rank++; ?></td>
                            <td><code><?php echo esc_html($country->country_code); ?></code></td>
                            <td><?php echo esc_html($country->country_name); ?></td>
                            <td><?php echo number_format($country->total_requests); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php
}

/**
 * Technology Tab
 */
function render_technology_tab($zone_id) {
    $period = isset($_GET['period']) ? $_GET['period'] : '7days';
    
    $periods = array(
        '7days' => array('label' => 'Last 7 Days', 'start' => date('Y-m-d', strtotime('-7 days')), 'end' => date('Y-m-d')),
        '30days' => array('label' => 'Last 30 Days', 'start' => date('Y-m-d', strtotime('-30 days')), 'end' => date('Y-m-d'))
    );
    
    $current_period = $periods[$period] ?? $periods['7days'];
    $browsers = CF_Database::get_top_browsers($zone_id, $current_period['start'], $current_period['end'], 20);
    
    // Calculate total for percentages
    $total_pageviews = array_sum(array_column($browsers, 'total_pageviews'));
    
    ?>
    <div class="cf-period-selector">
        <a href="?page=cf-analytics&tab=technology&period=7days" class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Last 7 Days</a>
        <a href="?page=cf-analytics&tab=technology&period=30days" class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Last 30 Days</a>
    </div>
    
    <div class="cf-table-container">
        <h2>Browser Statistics</h2>
        
        <?php if (empty($browsers)): ?>
            <div class="notice notice-info inline">
                <p>No browser data available yet. The daily sync will populate this data starting tomorrow.</p>
            </div>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 60px;">#</th>
                        <th>Browser</th>
                        <th style="width: 150px;">Page Views</th>
                        <th style="width: 150px;">Percentage</th>
                        <th>Usage</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $rank = 1; ?>
                    <?php foreach ($browsers as $browser): ?>
                        <?php 
                        $percentage = $total_pageviews > 0 ? ($browser->total_pageviews / $total_pageviews) * 100 : 0;
                        ?>
                        <tr>
                            <td><?php echo $rank++; ?></td>
                            <td><?php echo esc_html($browser->browser_name); ?></td>
                            <td><?php echo number_format($browser->total_pageviews); ?></td>
                            <td><?php echo round($percentage, 1); ?>%</td>
                            <td>
                                <div style="background: #e0e0e0; height: 20px; border-radius: 4px; overflow: hidden;">
                                    <div style="background: #667eea; height: 100%; width: <?php echo $percentage; ?>%;"></div>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php
}


/**
 * Traffic Analysis Tab
 */

/**
 * Traffic Analysis Tab
 */
function render_traffic_tab($zone_id) {
    $period = isset($_GET['period']) ? $_GET['period'] : 'today';
    
    $periods = array(
        'today' => array('label' => 'Today', 'start' => gmdate('Y-m-d'), 'end' => gmdate('Y-m-d')),
        'yesterday' => array('label' => 'Yesterday', 'start' => gmdate('Y-m-d', strtotime('-1 day')), 'end' => gmdate('Y-m-d', strtotime('-1 day'))),
        '7days' => array('label' => 'Last 7 Days', 'start' => gmdate('Y-m-d', strtotime('-7 days')), 'end' => gmdate('Y-m-d')),
        '30days' => array('label' => 'Last 30 Days', 'start' => gmdate('Y-m-d', strtotime('-30 days')), 'end' => gmdate('Y-m-d')),
        'year' => array('label' => 'Last Year', 'start' => gmdate('Y-m-d', strtotime('-365 days')), 'end' => gmdate('Y-m-d')),
        'alltime' => array('label' => 'All Time', 'start' => '2020-01-01', 'end' => gmdate('Y-m-d'))
    );
    
    $current_period = $periods[$period] ?? $periods['7days'];
    
    // Fetch data from database
    $status_codes = CF_Database::get_status_codes($zone_id, $current_period['start'], $current_period['end']);
    $device_types = CF_Database::get_device_types($zone_id, $current_period['start'], $current_period['end']);
    
    $has_data = !empty($status_codes) || !empty($device_types);
    
    ?>
    <div class="cf-period-selector">
        <a href="?page=cf-analytics&tab=traffic&period=today" class="button <?php echo $period === 'today' ? 'button-primary' : ''; ?>">Today</a>
        <a href="?page=cf-analytics&tab=traffic&period=yesterday" class="button <?php echo $period === 'yesterday' ? 'button-primary' : ''; ?>">Yesterday</a>
        <a href="?page=cf-analytics&tab=traffic&period=7days" class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Last 7 Days</a>
        <a href="?page=cf-analytics&tab=traffic&period=30days" class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Last 30 Days</a>
        <a href="?page=cf-analytics&tab=traffic&period=year" class="button <?php echo $period === 'year' ? 'button-primary' : ''; ?>">Last Year</a>
        <a href="?page=cf-analytics&tab=traffic&period=alltime" class="button <?php echo $period === 'alltime' ? 'button-primary' : ''; ?>">All Time</a>
    </div>
    
    <?php if (!$has_data): ?>
        <div class="notice notice-info inline">
            <p>No traffic data available yet. Click "Sync All Daily Data" in Settings to populate this data.</p>
        </div>
    <?php else: ?>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 20px;">
            <!-- HTTP Status Codes -->
            <?php if (!empty($status_codes)): ?>
                <div class="cf-table-container">
                    <h2>HTTP Status Codes</h2>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th style="width: 120px;">Status Code</th>
                                <th>Requests</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($status_codes as $item): ?>
                            <tr>
                                <td><strong><?php echo esc_html($item->status_code); ?></strong></td>
                                <td><?php echo number_format($item->total_requests); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
            
            <!-- Device Types -->
            <?php if (!empty($device_types)): ?>
                <div class="cf-table-container">
                    <h2>Device Types</h2>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th>Device Type</th>
                                <th>Requests</th>
                                <th>Percentage</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $total_device_requests = array_sum(array_map(function($d) { return $d->total_requests; }, $device_types));
                            foreach ($device_types as $item): 
                                $percentage = $total_device_requests > 0 ? round(($item->total_requests / $total_device_requests) * 100, 1) : 0;
                            ?>
                            <tr>
                                <td><strong><?php echo esc_html(ucfirst($item->device_type)); ?></strong></td>
                                <td><?php echo number_format($item->total_requests); ?></td>
                                <td><?php echo $percentage; ?>%</td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    <?php endif; ?>
    <?php
}

function render_bots_tab($zone_id) {
    $period = isset($_GET['period']) ? $_GET['period'] : 'today';
    
    $periods = array(
        'today' => array('label' => 'Today', 'start' => gmdate('Y-m-d'), 'end' => gmdate('Y-m-d')),
        'yesterday' => array('label' => 'Yesterday', 'start' => gmdate('Y-m-d', strtotime('-1 day')), 'end' => gmdate('Y-m-d', strtotime('-1 day'))),
        '7days' => array('label' => 'Last 7 Days', 'start' => gmdate('Y-m-d', strtotime('-7 days')), 'end' => gmdate('Y-m-d')),
        '30days' => array('label' => 'Last 30 Days', 'start' => gmdate('Y-m-d', strtotime('-30 days')), 'end' => gmdate('Y-m-d')),
        'year' => array('label' => 'Last Year', 'start' => gmdate('Y-m-d', strtotime('-365 days')), 'end' => gmdate('Y-m-d')),
        'alltime' => array('label' => 'All Time', 'start' => '2020-01-01', 'end' => gmdate('Y-m-d'))
    );
    
    $current_period = $periods[$period] ?? $periods['7days'];
    
    // Fetch user agents from database
    $user_agents = CF_Database::get_user_agents($zone_id, $current_period['start'], $current_period['end'], 50);
    
    ?>
    <div class="cf-period-selector">
        <a href="?page=cf-analytics&tab=bots&period=today" class="button <?php echo $period === 'today' ? 'button-primary' : ''; ?>">Today</a>
        <a href="?page=cf-analytics&tab=bots&period=yesterday" class="button <?php echo $period === 'yesterday' ? 'button-primary' : ''; ?>">Yesterday</a>
        <a href="?page=cf-analytics&tab=bots&period=7days" class="button <?php echo $period === '7days' ? 'button-primary' : ''; ?>">Last 7 Days</a>
        <a href="?page=cf-analytics&tab=bots&period=30days" class="button <?php echo $period === '30days' ? 'button-primary' : ''; ?>">Last 30 Days</a>
        <a href="?page=cf-analytics&tab=bots&period=year" class="button <?php echo $period === 'year' ? 'button-primary' : ''; ?>">Last Year</a>
        <a href="?page=cf-analytics&tab=bots&period=alltime" class="button <?php echo $period === 'alltime' ? 'button-primary' : ''; ?>">All Time</a>
    </div>
    
    <?php if (empty($user_agents)): ?>
        <div class="notice notice-info inline">
            <p>No user agent data available yet. Click "Sync All Daily Data" in Settings to populate this data.</p>
        </div>
    <?php else: ?>
        <div class="cf-table-container">
            <h2>Top User Agents</h2>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 60px;">#</th>
                        <th>User Agent</th>
                        <th style="width: 100px;">Type</th>
                        <th style="width: 120px;">Requests</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $rank = 1;
                    foreach ($user_agents as $ua): 
                        $agent = $ua->user_agent;
                        
                        // Detect type
                        $type = 'Browser';
                        $color = '#667eea';
                        
                        if (stripos($agent, 'bot') !== false || stripos($agent, 'crawler') !== false || stripos($agent, 'spider') !== false) {
                            $type = 'Bot';
                            $color = '#f093fb';
                        }
                        
                        if (stripos($agent, 'setup-config') !== false || stripos($agent, 'wp-admin') !== false || stripos($agent, 'wordpress') !== false) {
                            $type = 'Attack';
                            $color = '#f87171';
                        }
                        
                        // Truncate long user agents
                        $display_agent = strlen($agent) > 100 ? substr($agent, 0, 97) . '...' : $agent;
                    ?>
                    <tr>
                        <td><?php echo $rank++; ?></td>
                        <td><code style="font-size: 11px; word-break: break-all;"><?php echo esc_html($display_agent); ?></code></td>
                        <td>
                            <span style="display: inline-block; padding: 3px 8px; border-radius: 3px; font-size: 11px; font-weight: 600; background: <?php echo $color; ?>; color: white;">
                                <?php echo $type; ?>
                            </span>
                        </td>
                        <td><?php echo number_format($ua->total_requests); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
    <?php
}
