<?php
/**
 * WordPress Admin Bar Integration
 */

if (!defined('ABSPATH')) {
    exit;
}

class CF_Analytics_Admin_Bar {
    
    public function __construct() {
        add_action('admin_bar_menu', array($this, 'add_admin_bar_item'), 100);
        add_action('wp_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_styles'));
    }
    
    /**
     * Add admin bar item
     */
    public function add_admin_bar_item($wp_admin_bar) {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $settings = get_option('cf_analytics_settings', array());
        if (empty($settings['zone_id'])) {
            return;
        }
        
        $zone_id = $settings['zone_id'];
        
        // Get today's stats (use UTC since CF stores in UTC)
        $start_date = gmdate('Y-m-d') . ' 00:00:00';
        $end_date = gmdate('Y-m-d H:i:s');
        $totals = CF_Database::get_period_totals($zone_id, $start_date, $end_date);
        
        // Get top 5 content URLs today
        $top_urls = CF_Database::get_top_content_urls($zone_id, gmdate('Y-m-d'), gmdate('Y-m-d'), 5);
        
        // Main menu item
        $wp_admin_bar->add_node(array(
            'id' => 'cf-analytics',
            'title' => '<span class="cf-cloud-icon">☁</span> Analytics',
            'href' => admin_url('admin.php?page=cf-analytics'),
        ));
        
        // Add submenu items with stats
        $wp_admin_bar->add_node(array(
            'parent' => 'cf-analytics',
            'id' => 'cf-analytics-stats-header',
            'title' => 'Today\'s Stats',
            'href' => false,
            'meta' => array('class' => 'cf-stats-header')
        ));
        
        $wp_admin_bar->add_node(array(
            'parent' => 'cf-analytics',
            'id' => 'cf-analytics-requests',
            'title' => 'Requests: ' . CF_API::format_number($totals->total_requests ?? 0),
            'href' => admin_url('admin.php?page=cf-analytics'),
        ));
        
        $wp_admin_bar->add_node(array(
            'parent' => 'cf-analytics',
            'id' => 'cf-analytics-visitors',
            'title' => 'Visitors: ' . CF_API::format_number($totals->total_visitors ?? 0),
            'href' => admin_url('admin.php?page=cf-analytics'),
        ));
        
        $wp_admin_bar->add_node(array(
            'parent' => 'cf-analytics',
            'id' => 'cf-analytics-bandwidth',
            'title' => 'Bandwidth: ' . CF_API::format_bytes($totals->total_bandwidth ?? 0),
            'href' => admin_url('admin.php?page=cf-analytics'),
        ));
        
        $wp_admin_bar->add_node(array(
            'parent' => 'cf-analytics',
            'id' => 'cf-analytics-cache',
            'title' => 'Cache Hit: ' . round($totals->cache_hit_rate ?? 0, 1) . '%',
            'href' => admin_url('admin.php?page=cf-analytics'),
        ));
        
        if (!empty($top_urls)) {
            $wp_admin_bar->add_node(array(
                'parent' => 'cf-analytics',
                'id' => 'cf-analytics-top-header',
                'title' => 'Top Content',
                'href' => false,
                'meta' => array('class' => 'cf-stats-header')
            ));
            
            foreach (array_slice($top_urls, 0, 5) as $idx => $url) {
                // Get title for content
                $title = $url->url_path;
                $post_id = url_to_postid($url->url_path);
                if ($post_id) {
                    $title = get_the_title($post_id);
                } elseif ($url->url_path === '/' || $url->url_path === '') {
                    $title = 'Homepage';
                }
                
                // Truncate if needed
                if (strlen($title) > 30) {
                    $title = substr($title, 0, 27) . '...';
                }
                
                $wp_admin_bar->add_node(array(
                    'parent' => 'cf-analytics',
                    'id' => 'cf-analytics-url-' . $idx,
                    'title' => $title . ' (' . CF_API::format_number($url->total_requests) . ')',
                    'href' => admin_url('admin.php?page=cf-analytics&tab=urls'),
                ));
            }
        }
    }
    
    /**
     * Enqueue styles
     */
    public function enqueue_styles() {
        if (!is_admin_bar_showing() || !current_user_can('manage_options')) {
            return;
        }
        
        $settings = get_option('cf_analytics_settings', array());
        if (empty($settings['zone_id'])) {
            return;
        }
        
        ?>
        <style>
        /* Admin Bar Item */
        #wpadminbar #wp-admin-bar-cf-analytics > .ab-item {
            padding: 0 12px;
        }
        #wpadminbar #wp-admin-bar-cf-analytics .cf-cloud-icon {
            color: #f38020;
            font-size: 16px;
            display: inline-block;
            vertical-align: middle;
            margin-right: 5px;
            line-height: 1;
        }
        #wpadminbar #wp-admin-bar-cf-analytics:hover .cf-cloud-icon {
            color: #ff9f4d;
        }
        
        /* Submenu */
        #wpadminbar #wp-admin-bar-cf-analytics .ab-submenu {
            background: #23282d;
            min-width: 240px;
        }
        
        /* Headers */
        #wpadminbar .cf-stats-header .ab-item {
            background: #f38020 !important;
            color: #fff !important;
            font-weight: 600;
            font-size: 11px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            padding: 8px 12px !important;
        }
        #wpadminbar .cf-stats-header:hover .ab-item {
            background: #f38020 !important;
        }
        
        /* Stats Items */
        #wpadminbar #wp-admin-bar-cf-analytics .ab-submenu > li:not(.cf-stats-header) .ab-item {
            color: #eee;
            padding: 8px 12px;
            font-size: 13px;
        }
        #wpadminbar #wp-admin-bar-cf-analytics .ab-submenu > li:not(.cf-stats-header):hover .ab-item {
            background: #32373c;
            color: #00b9eb;
        }
        </style>
        <?php
    }
}

new CF_Analytics_Admin_Bar();
